<?php
/**
 * Auth Controller
 * Gerencia login, logout e informações do usuário
 */

namespace Controllers;

use Core\Request;
use Core\Response;
use Middleware\Auth;

class AuthController
{
    /**
     * POST /login - Autenticar usuário
     */
    public static function login(): void
    {
        $username = Request::input('username', '');
        $password = Request::input('password', '');

        if (!$username || !$password) {
            Response::error('Username and password required', 400);
        }

        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE username = ? AND is_active = 1");
        $stmt->execute([$username]);
        $user = $stmt->fetch();

        if (!$user || !password_verify($password, $user['password_hash'])) {
            logAudit('login_failed', null, "Username: {$username}");
            Response::error('Invalid credentials', 401);
        }

        try {
            // Criar sessão no banco
            $sessionId = generateSecureToken(64);
            $expiresAt = date('Y-m-d H:i:s', strtotime('+24 hours'));

            $stmt = $pdo->prepare("
                INSERT INTO admin_sessions (id, user_id, ip_address, user_agent, expires_at)
                VALUES (?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $sessionId,
                $user['id'],
                Request::ip(),
                Request::userAgent(),
                $expiresAt
            ]);

            // Atualizar last_login
            $pdo->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = ?")->execute([$user['id']]);

            // Criar sessão PHP (se sessão está ativa)
            if (session_status() === PHP_SESSION_ACTIVE) {
                Auth::createSession($user);
            }

            // Log de auditoria (ignorar erros)
            try {
                logAudit('login_success', null, "User: {$username}", Request::ip());
            } catch (\Exception $e) {
                // Ignorar erro de log
            }

            Response::success([
                'token' => $sessionId,
                'user' => [
                    'username' => $user['username'],
                    'role' => $user['role']
                ],
                'expires_at' => $expiresAt
            ]);
        } catch (\Exception $e) {
            Response::error('Login failed: ' . $e->getMessage(), 500);
        }
    }

    /**
     * POST /logout - Encerrar sessão
     */
    public static function logout(): void
    {
        Auth::destroySession();
        Response::success(null, 'Logged out');
    }

    /**
     * GET /me - Informações do usuário logado
     * OTIMIZADO: Query única, removidos checks de schema
     */
    public static function me(): void
    {
        $currentUser = Auth::require();

        $pdo = getDBConnection();

        // Query otimizada com GROUP_CONCAT para permissões
        $stmt = $pdo->prepare("
            SELECT u.id, u.username, u.email, u.role, u.site_id, u.last_login,
                   s.name as site_name,
                   GROUP_CONCAT(DISTINCT p.permission) as permissions_csv
            FROM admin_users u
            LEFT JOIN sites s ON s.id = u.site_id
            LEFT JOIN site_user_permissions p ON p.user_id = u.id
            WHERE u.id = ?
            GROUP BY u.id
        ");
        $stmt->execute([$currentUser['id']]);
        $user = $stmt->fetch();

        if (!$user) {
            Response::error('User not found', 404);
        }

        $user['is_super_admin'] = Auth::isSuperAdmin($currentUser);

        // Processar permissões
        if (!empty($user['permissions_csv'])) {
            $user['permissions'] = explode(',', $user['permissions_csv']);
        } elseif (empty($user['site_id'])) {
            $user['permissions'] = ['all'];
        } else {
            $user['permissions'] = [];
        }
        unset($user['permissions_csv']);

        Response::success($user);
    }

    /**
     * GET /permissions - Lista de permissões disponíveis
     */
    public static function listPermissions(): void
    {
        Response::success([
            ['key' => 'view_dashboard', 'label' => 'Ver Painel', 'description' => 'Visualizar dashboard do site'],
            ['key' => 'view_events', 'label' => 'Ver Eventos', 'description' => 'Visualizar eventos/tracking'],
            ['key' => 'view_stats', 'label' => 'Ver Estatísticas', 'description' => 'Visualizar estatísticas'],
            ['key' => 'edit_pix', 'label' => 'Editar PIX', 'description' => 'Editar configuração PIX'],
            ['key' => 'view_pix_history', 'label' => 'Ver Histórico PIX', 'description' => 'Ver histórico de alterações PIX'],
            ['key' => 'manage_site', 'label' => 'Gerenciar Site', 'description' => 'Editar informações do site']
        ]);
    }

    /**
     * PUT /permissions - Atualizar permissões de um usuário
     * OTIMIZADO: Removido SHOW TABLES, batch insert
     */
    public static function updatePermissions(): void
    {
        $currentUser = Auth::require();

        // Apenas super_admin pode alterar permissões
        if (!Auth::isSuperAdmin($currentUser)) {
            Response::forbidden('Only super admins can update permissions');
        }

        $userId = Request::getInt('user_id');
        $data = Request::body();

        if (!$userId) {
            Response::error('User ID required', 400);
        }

        $siteId = $data['site_id'] ?? null;
        $permissions = $data['permissions'] ?? [];

        if (!$siteId) {
            Response::error('Site ID required', 400);
        }

        if (!is_array($permissions)) {
            Response::error('Permissions must be an array', 400);
        }

        $pdo = getDBConnection();

        // Verificar se o usuário existe
        $stmt = $pdo->prepare("SELECT id, username FROM admin_users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();

        if (!$user) {
            Response::notFound('User not found');
        }

        // Remover permissões antigas do usuário para o site
        $pdo->prepare("DELETE FROM site_user_permissions WHERE user_id = ? AND site_id = ?")->execute([$userId, $siteId]);

        // Inserir novas permissões (batch insert)
        if (!empty($permissions)) {
            $values = [];
            $params = [];
            foreach ($permissions as $permission) {
                $values[] = "(?, ?, ?)";
                $params[] = $userId;
                $params[] = $siteId;
                $params[] = $permission;
            }
            $sql = "INSERT INTO site_user_permissions (user_id, site_id, permission) VALUES " . implode(',', $values);
            $pdo->prepare($sql)->execute($params);
        }

        Response::success([
            'user_id' => $userId,
            'site_id' => $siteId,
            'permissions' => $permissions,
            'message' => 'Permissions updated successfully'
        ]);
    }
}
