<?php
/**
 * PIX Controller
 * Gerencia configurações e estatísticas PIX
 */

namespace Controllers;

use Core\Request;
use Core\Response;
use Middleware\Auth;

class PixController
{
    /**
     * GET /pix-configs - Listar todas as configurações PIX
     */
    public static function listConfigs(): void
    {
        Auth::require();
        $pdo = getDBConnection();
        
        $stmt = $pdo->query("
            SELECT p.*, s.name as site_name 
            FROM pix_config p 
            JOIN sites s ON s.id = p.site_id 
            ORDER BY s.name ASC, p.state ASC
        ");
        $configs = $stmt->fetchAll();

        foreach ($configs as &$c) {
            $c['pix_key'] = decryptData($c['pix_key']);
        }

        Response::success($configs);
    }

    /**
     * GET /pix-config?site_id={id} - Obter configuração PIX do site
     * Também suporta X-API-Key para acesso público pelos sites
     */
    public static function getConfig(): void
    {
        // Verificar se é acesso via API Key (público)
        $apiKey = Request::apiKey();
        $state = Request::get('state', 'BR');

        if ($apiKey) {
            self::getConfigByApiKey($apiKey, $state);
            return;
        }

        // Acesso autenticado
        $currentUser = Auth::require();
        $siteId = (int)Request::get('site_id', 0);

        if (!$siteId && !empty($currentUser['site_id'])) {
            $siteId = (int)$currentUser['site_id'];
        }

        if (!$siteId) {
            Response::error('Site ID required', 400);
        }

        Auth::requireSiteAccess($siteId);

        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT * FROM pix_config WHERE site_id = ? AND state = ?");
        $stmt->execute([$siteId, $state]);
        $config = $stmt->fetch();

        // Se não achar por estado, tenta buscar a global 'BR'
        if (!$config && $state !== 'BR') {
            $stmt = $pdo->prepare("SELECT * FROM pix_config WHERE site_id = ? AND state = 'BR'");
            $stmt->execute([$siteId]);
            $config = $stmt->fetch();
        }

        if ($config) {
            $config['pix_key'] = decryptData($config['pix_key']);
        }

        Response::success($config ?: null);
    }

    private static function getConfigByApiKey(string $apiKey, string $state = 'BR'): void
    {
        $pdo = getDBConnection();

        $stmt = $pdo->prepare("SELECT id FROM sites WHERE api_key = ? AND is_active = 1");
        $stmt->execute([$apiKey]);
        $site = $stmt->fetch();

        if (!$site) Response::error('Invalid API Key', 401);

        $stmt = $pdo->prepare("SELECT * FROM pix_config WHERE site_id = ? AND state = ?");
        $stmt->execute([$site['id'], $state]);
        $config = $stmt->fetch();
        
        if (!$config && $state !== 'BR') {
            $stmt = $pdo->prepare("SELECT * FROM pix_config WHERE site_id = ? AND state = 'BR'");
            $stmt->execute([$site['id']]);
            $config = $stmt->fetch();
        }

        if ($config) {
            $config['pix_key'] = decryptData($config['pix_key']);
            Response::success([
                'pixKey' => $config['pix_key'],
                'pixKeyType' => $config['pix_key_type'],
                'receiverName' => $config['receiver_name'],
                'receiverCity' => $config['receiver_city']
            ]);
        } else {
            Response::success(null);
        }
    }

    /**
     * POST /pix-config?site_id={id} - Salvar configuração PIX
     */
    public static function saveConfig(): void
    {
        $currentUser = Auth::require();
        
        $data = Request::body();
        $siteId = Request::getInt('site_id') ?: ($data['site_id'] ?? 0);

        // Se for usuário de site específico, forçar o site_id dele
        if (!Auth::isSuperAdmin($currentUser) && !empty($currentUser['site_id'])) {
            $siteId = (int)$currentUser['site_id'];
        }

        if (!$siteId) {
            Response::error('Site ID required', 400);
        }

        // Verificar acesso
        Auth::requireSiteAccess($siteId);

        // Verificar permissão para editar PIX
        if (!Auth::hasPermission('edit_pix', $currentUser)) {
            Response::forbidden('Permissão negada para editar PIX');
        }

        $pixKey = $data['pix_key'] ?? Request::input('pix_key', '');
        $receiverName = substr($data['receiver_name'] ?? Request::input('receiver_name', 'PAGAMENTO'), 0, 25);
        $receiverCity = substr($data['receiver_city'] ?? Request::input('receiver_city', 'SAO PAULO'), 0, 15);
        $state = $data['state'] ?? Request::input('state', 'BR');
        
        // Validar pix_key_type contra ENUM do banco
        $allowedTypes = ['cpf', 'cnpj', 'email', 'phone', 'random'];
        $pixKeyType = $data['pix_key_type'] ?? Request::input('pix_key_type', 'random');
        if (!in_array($pixKeyType, $allowedTypes)) {
            $pixKeyType = 'random';
        }

        if (!$pixKey) {
            Response::error('PIX key required', 400);
        }

        $pdo = getDBConnection();

        // Buscar config atual para histórico
        $stmt = $pdo->prepare("SELECT * FROM pix_config WHERE site_id = ? AND state = ?");
        $stmt->execute([$siteId, $state]);
        $oldConfig = $stmt->fetch();

        // Criptografar chave
        $encryptedKey = encryptData($pixKey);

        // Inserir ou atualizar
        $stmt = $pdo->prepare("
            INSERT INTO pix_config (site_id, state, pix_key, pix_key_type, receiver_name, receiver_city)
            VALUES (?, ?, ?, ?, ?, ?)
            ON DUPLICATE KEY UPDATE
                state = VALUES(state),
                pix_key = VALUES(pix_key),
                pix_key_type = VALUES(pix_key_type),
                receiver_name = VALUES(receiver_name),
                receiver_city = VALUES(receiver_city),
                updated_at = NOW()
        ");
        $stmt->execute([$siteId, $state, $encryptedKey, $pixKeyType, $receiverName, $receiverCity]);

        // Registrar histórico
        $stmt = $pdo->prepare("
            INSERT INTO pix_config_history (
                site_id, change_type, 
                old_pix_key, new_pix_key,
                old_receiver_name, new_receiver_name,
                old_receiver_city, new_receiver_city,
                changed_by_ip, changed_by_user_id
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $siteId,
            $oldConfig ? 'update' : 'create',
            $oldConfig ? decryptData($oldConfig['pix_key']) : null,
            $pixKey,
            $oldConfig['receiver_name'] ?? null,
            $receiverName,
            $oldConfig['receiver_city'] ?? null,
            $receiverCity,
            Request::ip(),
            $currentUser['id']
        ]);

        logAudit('pix_config_updated', $siteId, "User: {$currentUser['username']}", Request::ip());

        Response::success(null, 'PIX config saved');
    }

    /**
     * GET /pix-history?site_id={id} - Histórico de alterações PIX
     */
    public static function getHistory(): void
    {
        $currentUser = Auth::require();
        $siteId = (int)Request::get('site_id', 0);

        // Se for usuário de site, forçar filtro pelo site dele
        if (!Auth::isSuperAdmin($currentUser) && !empty($currentUser['site_id'])) {
            $siteId = (int)$currentUser['site_id'];

            // Verificar permissão
            if (!Auth::hasPermission('view_pix_history', $currentUser)) {
                Response::forbidden('Permissão negada para ver histórico PIX');
            }
        }

        $pdo = getDBConnection();
        $sql = "SELECT h.*, u.username as changed_by_user, s.name as site_name
                FROM pix_config_history h
                LEFT JOIN admin_users u ON h.changed_by_user_id = u.id
                LEFT JOIN sites s ON s.id = h.site_id";

        if ($siteId) {
            $sql .= " WHERE h.site_id = ?";
            $stmt = $pdo->prepare($sql . " ORDER BY h.created_at DESC LIMIT 100");
            $stmt->execute([$siteId]);
        } else {
            $stmt = $pdo->query($sql . " ORDER BY h.created_at DESC LIMIT 100");
        }

        $history = $stmt->fetchAll();

        Response::success($history);
    }

    /**
     * GET /pix-stats?site_id={id} - Estatísticas de PIX e todos os tipos de eventos
     */
    public static function getStats(): void
    {
        $currentUser = Auth::require();
        $siteId = (int)Request::get('site_id', 0);

        // Se for usuário de site, forçar filtro pelo site dele
        if (!Auth::isSuperAdmin($currentUser) && !empty($currentUser['site_id'])) {
            $siteId = (int)$currentUser['site_id'];
        }

        $pdo = getDBConnection();
        $params = [];
        $whereBase = "1=1";

        if ($siteId) {
            $whereBase .= " AND site_id = ?";
            $params[] = $siteId;
        }

        // Contagem por tipo de evento (hoje e total)
        $stmt = $pdo->prepare("
            SELECT
                event_type,
                COUNT(*) as total,
                SUM(CASE WHEN DATE(created_at) = CURDATE() THEN 1 ELSE 0 END) as today,
                COALESCE(SUM(valor), 0) as valor_total,
                COALESCE(SUM(CASE WHEN DATE(created_at) = CURDATE() THEN valor ELSE 0 END), 0) as valor_hoje
            FROM events
            WHERE {$whereBase}
            GROUP BY event_type
            ORDER BY total DESC
        ");
        $stmt->execute($params);
        $eventStats = $stmt->fetchAll(\PDO::FETCH_ASSOC);

        // Tipos de eventos PIX para contagem (apenas geração, não cópia)
        $pixEventTypes = "('pix_gerado', 'pix_generation_success', 'pix_generated', 'pix_iniciado')";

        // Total PIX gerados
        $stmt = $pdo->prepare("SELECT COUNT(*), COALESCE(SUM(valor), 0) FROM events WHERE {$whereBase} AND event_type IN {$pixEventTypes}");
        $stmt->execute($params);
        $pixData = $stmt->fetch(\PDO::FETCH_NUM);
        $totalPix = $pixData[0];
        $totalValue = $pixData[1];

        // PIX de hoje
        $stmt = $pdo->prepare("SELECT COUNT(*), COALESCE(SUM(valor), 0) FROM events WHERE {$whereBase} AND event_type IN {$pixEventTypes} AND DATE(created_at) = CURDATE()");
        $stmt->execute($params);
        $pixTodayData = $stmt->fetch(\PDO::FETCH_NUM);
        $pixToday = $pixTodayData[0];
        $valorHoje = $pixTodayData[1];

        // IPs únicos com PIX
        $stmt = $pdo->prepare("SELECT COUNT(DISTINCT ip_address) FROM events WHERE {$whereBase} AND event_type IN {$pixEventTypes}");
        $stmt->execute($params);
        $uniqueIps = $stmt->fetchColumn();

        // Total de eventos
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE {$whereBase}");
        $stmt->execute($params);
        $totalEvents = $stmt->fetchColumn();

        // Eventos hoje
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE {$whereBase} AND DATE(created_at) = CURDATE()");
        $stmt->execute($params);
        $eventsToday = $stmt->fetchColumn();

        Response::success([
            'total_pix' => (int)$totalPix,
            'pix_today' => (int)$pixToday,
            'total_value' => round((float)$totalValue, 2),
            'valor_hoje' => round((float)$valorHoje, 2),
            'unique_ips' => (int)$uniqueIps,
            'total_events' => (int)$totalEvents,
            'events_today' => (int)$eventsToday,
            'by_type' => $eventStats
        ]);
    }
}
