<?php
namespace Controllers;

use Core\Request;
use Core\Response;
use Middleware\Auth;

class SettingsController
{
    private static $settingsFile = __DIR__ . '/../../config/settings.json';

    public static function get(): void
    {
        Auth::requireSuperAdmin();
        
        if (!file_exists(self::$settingsFile)) {
            Response::success([
                'capsolver_key' => '',
                'proxy_host' => 'gate.nodemaven.com:8080',
                'proxy_user' => '',
                'proxy_pass' => '',
                'api_dh_key' => ''
            ]);
            return;
        }

        $data = json_decode(file_get_contents(self::$settingsFile), true);
        Response::success($data);
    }

    public static function save(): void
    {
        Auth::requireSuperAdmin();
        $data = Request::body();

        if (empty($data)) {
            Response::error('Dados inválidos', 400);
        }

        // Preservar o saldo atual se não enviado
        $current = self::getInternal();
        if (!isset($data['last_captcha_balance'])) {
            $data['last_captcha_balance'] = $current['last_captcha_balance'] ?? '0.0000';
        }

        // Salvar no arquivo JSON
        file_put_contents(self::$settingsFile, json_encode($data, JSON_PRETTY_PRINT));

        // Registrar Log
        logAudit('global_settings_updated', null, "Configurações globais atualizadas", Request::ip());

        Response::success(null, 'Configurações salvas com sucesso');
    }

    public static function updateCaptchaBalance(): void
    {
        // Esta rota é chamada internamente pelo servidor Node
        $data = Request::body();
        $balance = $data['balance'] ?? null;

        if ($balance === null) {
            Response::error('Balance required', 400);
        }

        $settings = self::getInternal();
        $settings['last_captcha_balance'] = (string)$balance;
        
        file_put_contents(self::$settingsFile, json_encode($settings, JSON_PRETTY_PRINT));
        Response::success(null, 'Balance updated');
    }

    public static function testCaptcha(): void
    {
        Auth::requireSuperAdmin();
        $settings = self::getInternal();
        $key = $settings['capsolver_key'] ?? '';

        if (!$key) Response::error('Chave CapSolver não configurada', 400);

        $ch = curl_init('https://api.capsolver.com/getBalance');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode(['clientKey' => $key]));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        $res = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);

        $pdo = \getDBConnection();

        if ($error) {
            $stmt = $pdo->prepare("INSERT INTO captcha_logs (status, error_message) VALUES ('failed', ?)");
            $stmt->execute([$error]);
            Response::error('Erro na conexão: ' . $error, 500);
        }

        $data = json_decode($res, true);
        if (isset($data['balance'])) {
            $balance = (float)$data['balance'];
            
            // Log do teste
            $stmt = $pdo->prepare("INSERT INTO captcha_logs (balance, status) VALUES (?, 'success')");
            $stmt->execute([$balance]);

            // Atualiza o cache do saldo nas configurações
            $settings['last_captcha_balance'] = (string)$balance;
            file_put_contents(self::$settingsFile, json_encode($settings, JSON_PRETTY_PRINT));

            Response::success(['balance' => $balance], 'Saldo recuperado com sucesso');
        } else {
            $errDesc = $data['errorDescription'] ?? 'Erro desconhecido';
            $stmt = $pdo->prepare("INSERT INTO captcha_logs (status, error_message) VALUES ('failed', ?)");
            $stmt->execute([$errDesc]);
            Response::error('Falha na API: ' . $errDesc, 500);
        }
    }

    public static function testProxy(): void
    {
        Auth::requireSuperAdmin();
        $settings = self::getInternal();
        
        $host = $settings['proxy_host'] ?? '';
        $user = $settings['proxy_user'] ?? '';
        $pass = $settings['proxy_pass'] ?? '';

        if (!$host || !$user || !$pass) Response::error('Proxy não configurada corretamente', 400);

        // Gerar uma sessão aleatória para o teste
        $sessionId = 'test_' . bin2hex(random_bytes(4));
        $sessionUser = $user . '-session-' . $sessionId;
        $proxyUrl = "http://$sessionUser:$pass@$host";

        $startTime = microtime(true);
        
        $ch = curl_init('https://api.ipify.org?format=json');
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_PROXY, $proxyUrl);
        curl_setopt($ch, CURLOPT_TIMEOUT, 15);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $res = curl_exec($ch);
        $error = curl_error($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        $latency = (int)((microtime(true) - $startTime) * 1000);
        $pdo = \getDBConnection();

        if ($error || $httpCode !== 200) {
            // Registrar falha
            $stmt = $pdo->prepare("INSERT INTO proxy_logs (ip_address, gateway, session_id, status, latency_ms) VALUES (?, ?, ?, 'failed', ?)");
            $stmt->execute(['0.0.0.0', $host, $sessionId, $latency]);
            Response::error('Erro na Proxy: ' . ($error ?: "HTTP $httpCode"), 500);
        }

        $data = json_decode($res, true);
        $ip = $data['ip'] ?? 'Unknown';

        if ($ip !== 'Unknown') {
            // Registrar Sucesso
            $stmt = $pdo->prepare("INSERT INTO proxy_logs (ip_address, gateway, session_id, status, latency_ms) VALUES (?, ?, ?, 'active', ?)");
            $stmt->execute([$ip, $host, $sessionId, $latency]);
            
            // Também registra como evento de uso para o contador global
            $stmtEv = $pdo->prepare("INSERT INTO events (site_id, event_type, ip_address, metadata) VALUES ((SELECT id FROM sites LIMIT 1), 'proxy_usage', ?, ?)");
            $stmtEv->execute([$ip, json_encode(['sessionId' => $sessionId, 'source' => 'test_button', 'latency' => $latency])]);

            Response::success(['ip' => $ip, 'latency' => $latency], 'Proxy funcionando corretamente');
        } else {
            Response::error('Resposta inválida do servidor de IP', 500);
        }
    }

    private static function getInternal(): array
    {
        if (!file_exists(self::$settingsFile)) return [];
        return json_decode(file_get_contents(self::$settingsFile), true) ?: [];
    }
}
