<?php
namespace Core;

class CSRF {
    /**
     * Gera um token CSRF e armazena na sessão
     * @return string O token gerado
     */
    public static function generate(): string {
        if (session_status() === PHP_SESSION_NONE) session_start();
        
        $token = bin2hex(random_bytes(32));
        $_SESSION['csrf_token'] = $token;
        return $token;
    }

    /**
     * Obtém o token atual da sessão ou gera um novo
     * @return string O token CSRF
     */
    public static function getToken(): string {
        if (session_status() === PHP_SESSION_NONE) session_start();
        
        if (empty($_SESSION['csrf_token'])) {
            return self::generate();
        }
        
        return $_SESSION['csrf_token'];
    }

    /**
     * Valida se o token enviado é o mesmo da sessão
     * @return bool True se for válido, False caso contrário
     */
    public static function validate(): bool {
        if (session_status() === PHP_SESSION_NONE) session_start();
        
        // Se a sessão já expirou ou não tem token, falha
        if (empty($_SESSION['csrf_token'])) return false;

        // Tenta pegar o token do cabeçalho ou do POST
        $submittedToken = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? $_POST['csrf_token'] ?? '';
        
        if (empty($submittedToken)) {
            // Se for JSON, tenta pegar de dentro do body
            $jsonBody = json_decode(file_get_contents('php://input'), true);
            $submittedToken = $jsonBody['csrf_token'] ?? '';
        }

        return hash_equals($_SESSION['csrf_token'], $submittedToken);
    }
}
