/**
 * PIX Module
 * Configuração PIX, geração de QR Code, histórico
 */
const PIX = {
    // Site atualmente selecionado para config PIX
    currentSiteId: null,
    
    /**
     * Auxiliar para definir valor de um elemento se ele existir
     */
    setVal(id, val) {
        const el = document.getElementById(id);
        if (el) el.value = val;
    },

    /**
     * Auxiliar para pegar valor de um elemento se ele existir
     */
    getVal(id, defaultVal = '') {
        const el = document.getElementById(id);
        return el ? el.value : defaultVal;
    },

    /**
     * Atualiza o seletor de site na página PIX
     */
    updateSiteSelector() {
        const sitesData = Sites.getData();
        const selector = document.getElementById('pix-site-selector');
        if (!selector) return;

        selector.innerHTML = '<option value="">-- Selecione um Site --</option>';
        sitesData.forEach(site => {
            selector.innerHTML += `<option value="${site.id}">${site.name} (${site.domain})</option>`;
        });

        // Preencher Seletor de Estados
        const stateSelector = document.getElementById('pix-state');
        if (stateSelector) {
            const states = [
                { uf: 'BR', name: 'Global (Todos)' }, { uf: 'AC', name: 'Acre' }, { uf: 'AL', name: 'Alagoas' }, { uf: 'AP', name: 'Amapá' },
                { uf: 'AM', name: 'Amazonas' }, { uf: 'BA', name: 'Bahia' }, { uf: 'CE', name: 'Ceará' }, { uf: 'DF', name: 'Distrito Federal' },
                { uf: 'ES', name: 'Espírito Santo' }, { uf: 'GO', name: 'Goiás' }, { uf: 'MA', name: 'Maranhão' }, { uf: 'MT', name: 'Mato Grosso' },
                { uf: 'MS', name: 'Mato Grosso do Sul' }, { uf: 'MG', name: 'Minas Gerais' }, { uf: 'PA', name: 'Pará' }, { uf: 'PB', name: 'Paraíba' },
                { uf: 'PR', name: 'Paraná' }, { uf: 'PE', name: 'Pernambuco' }, { uf: 'PI', name: 'Piauí' }, { uf: 'RJ', name: 'Rio de Janeiro' },
                { uf: 'RN', name: 'Rio Grande do Norte' }, { uf: 'RS', name: 'Rio Grande do Sul' }, { uf: 'RO', name: 'Rondônia' },
                { uf: 'RR', name: 'Roraima' }, { uf: 'SC', name: 'Santa Catarina' }, { uf: 'SP', name: 'São Paulo' }, { uf: 'SE', name: 'Sergipe' },
                { uf: 'TO', name: 'Tocantins' }
            ];
            stateSelector.innerHTML = states.map(s => `<option value="${s.uf}">${s.name}</option>`).join('');
            stateSelector.addEventListener('change', () => this.loadConfig(this.currentSiteId));
        }
        
        // Auto-selecionar para usuário de site
        const currentUser = Auth.getCurrentUser();
        if (currentUser && currentUser.site_id && !currentUser.is_super_admin) {
            selector.value = currentUser.site_id;
            selector.disabled = true;
            this.currentSiteId = currentUser.site_id;
            
            const site = Sites.getById(currentUser.site_id);
            const siteName = document.getElementById('pix-site-name');
            if (siteName && site) {
                siteName.textContent = `${site.name} (${site.domain})`;
            }
            
            document.getElementById('pix-select-alert')?.classList.add('hidden');
            document.getElementById('pix-config-form')?.classList.remove('hidden');
            document.getElementById('pix-test-section')?.classList.remove('hidden');
            this.loadConfig(currentUser.site_id);
        }
    },
    
    /**
     * Handler do change no seletor de site
     */
    async onSiteChange(siteId) {
        this.currentSiteId = siteId;
        
        if (!siteId) {
            document.getElementById('pix-select-alert')?.classList.remove('hidden');
            document.getElementById('pix-config-form')?.classList.add('hidden');
            document.getElementById('pix-test-section')?.classList.add('hidden');
            return;
        }
        
        document.getElementById('pix-select-alert')?.classList.add('hidden');
        document.getElementById('pix-config-form')?.classList.remove('hidden');
        document.getElementById('pix-test-section')?.classList.remove('hidden');
        
        // Nome do site
        const site = Sites.getById(siteId);
        const siteNameEl = document.getElementById('pix-site-name');
        if (siteNameEl) {
            siteNameEl.textContent = site ? `${site.name} (${site.domain})` : '-';
        }
        
        await this.loadConfig(siteId);
    },
    
    /**
     * Carrega configuração PIX de um site
     */
    async loadConfig(siteId) {
        if (!siteId) return;
        const state = this.getVal('pix-state', 'BR');
        try {
            const data = await Api.get(`pix-config&site_id=${siteId}&state=${state}`);
            
            if (data.status === 'success' && data.data) {
                this.setVal('pix-key', data.data.pix_key || '');
                this.setVal('pix-key-type', data.data.pix_key_type || 'random');
                this.setVal('pix-receiver-name', data.data.receiver_name || 'PAGAMENTO');
                this.setVal('pix-receiver-city', data.data.receiver_city || 'SAO PAULO');
            } else {
                this.setVal('pix-key', '');
                this.setVal('pix-key-type', 'random');
                this.setVal('pix-receiver-name', 'PAGAMENTO');
                this.setVal('pix-receiver-city', 'SAO PAULO');
            }
        } catch (error) {
            console.error('PIX config error:', error);
        }
    },
    
    /**
     * Carrega todas as configurações PIX do sistema
     */
    async loadAllConfigs() {
        const tbody = document.getElementById('pix-configs-table-body');
        if (!tbody) return;

        try {
            const res = await Api.get('pix-configs');
            if (res.status === 'success') {
                const configs = res.data || [];
                if (configs.length === 0) {
                    tbody.innerHTML = '<tr><td colspan="5" class="text-center py-4 opacity-50">Nenhuma chave configurada</td></tr>';
                    return;
                }

                tbody.innerHTML = configs.map(c => `
                    <tr class="hover:bg-white/5 transition-colors border-b border-white/5">
                        <td class="py-3"><span class="text-cyan-400 font-bold text-[10px]">${c.site_name}</span></td>
                        <td><span class="badge bg-white/10 text-white text-[9px]">${c.state || 'BR'}</span></td>
                        <td><span class="text-[9px] uppercase opacity-60">${c.pix_key_type}</span></td>
                        <td><code class="text-[10px] text-slate-300">${c.pix_key}</code></td>
                        <td class="text-end">
                            <button class="btn btn-sm btn-link text-cyan-400 p-0 me-2" onclick="PIX.editFromTable(${c.site_id}, '${c.state}')"><i class="fas fa-edit"></i></button>
                            <button class="btn btn-sm btn-link text-red-400 p-0" onclick="PIX.deleteConfig(${c.id})"><i class="fas fa-trash"></i></button>
                        </td>
                    </tr>
                `).join('');
            }
        } catch (error) {
            console.error('[PIX] Erro ao carregar chaves:', error);
        }
    },

    editFromTable(siteId, state) {
        this.currentSiteId = siteId;
        document.getElementById('pix-site-selector').value = siteId;
        document.getElementById('pix-state').value = state;
        this.onSiteChange(siteId);
        window.scrollTo({ top: 0, behavior: 'smooth' });
    },

    async deleteConfig(id) {
        if (!confirm('Deseja realmente remover esta configuração PIX?')) return;
        try {
            await Api.delete(`pix-config&id=${id}`);
            this.loadAllConfigs();
        } catch(e) {}
    },

    /**
     * Salva configuração PIX
     */
    async saveConfig(e) {
        if (e) e.preventDefault();
        
        if (!this.currentSiteId) {
            alert('Selecione um site primeiro');
            return;
        }
        
        const payload = {
            state: document.getElementById('pix-state')?.value || 'BR',
            pix_key: document.getElementById('pix-key')?.value || '',
            pix_key_type: document.getElementById('pix-key-type')?.value || 'random',
            receiver_name: document.getElementById('pix-receiver-name')?.value || 'PAGAMENTO',
            receiver_city: document.getElementById('pix-receiver-city')?.value || 'SAO PAULO'
        };
        
        console.log('[PIX] Salvando config:', payload);
        
        try {
            const data = await Api.post(`pix-config&site_id=${this.currentSiteId}`, payload);
            
            if (data.status === 'success') {
                UI.toast('Configuração PIX salva com sucesso!', 'success');
                this.loadAllConfigs(); // RECARREGA A LISTA
            } else {
                alert('Erro ao salvar: ' + (data.message || 'Erro desconhecido'));
            }
        } catch (error) {
            alert('Erro ao salvar configuração PIX');
            console.error(error);
        }
    },
    
    /**
     * Carrega histórico de alterações PIX
     */
    async loadHistory() {
        try {
            const data = await Api.get(`pix-history&site_id=${App.currentSiteId || ''}`);
            
            if (data.status === 'success') {
                const sitesData = Sites.getData();
                const tbody = document.getElementById('pix-history-body');
                if (!tbody) return;
                
                tbody.innerHTML = data.data.map(h => {
                    const time = h.created_at ? new Date(h.created_at).toLocaleString('pt-BR') : '-';
                    const site = sitesData.find(s => s.id == h.site_id);
                    
                    return `
                        <tr>
                            <td><small>${time}</small></td>
                            <td>${site?.name || 'Site #' + h.site_id}</td>
                            <td><span class="badge ${h.change_type === 'create' ? 'bg-success' : 'bg-warning'}">${h.change_type}</span></td>
                            <td><code>${h.old_pix_key || '-'}</code></td>
                            <td><code>${h.new_pix_key || '-'}</code></td>
                            <td>${h.changed_by_user || '-'}</td>
                            <td><small>${h.changed_by_ip || '-'}</small></td>
                        </tr>
                    `;
                }).join('') || '<tr><td colspan="7" class="text-center">Nenhum histórico</td></tr>';
            }
        } catch (error) {
            console.error('PIX History error:', error);
        }
    },
    
    /**
     * Gera código PIX EMV
     */
    generateCode(pixKey, receiverName, receiverCity, amount, description = '') {
        receiverName = receiverName.toUpperCase().substring(0, 25).normalize('NFD').replace(/[\u0300-\u036f]/g, '');
        receiverCity = receiverCity.toUpperCase().substring(0, 15).normalize('NFD').replace(/[\u0300-\u036f]/g, '');
        
        const merchantCategoryCode = '0000';
        const transactionCurrency = '986'; 
        const countryCode = 'BR';
        
        const tlv = (id, value) => {
            const len = value.length.toString().padStart(2, '0');
            return id + len + value;
        };
        
        let merchantAccountInfo = tlv('00', 'br.gov.bcb.pix');
        merchantAccountInfo += tlv('01', pixKey);
        if (description) {
            merchantAccountInfo += tlv('02', description.substring(0, 50));
        }
        
        let payload = '';
        payload += tlv('00', '01'); 
        payload += tlv('26', merchantAccountInfo); 
        payload += tlv('52', merchantCategoryCode);
        payload += tlv('53', transactionCurrency);
        
        if (amount > 0) {
            payload += tlv('54', amount.toFixed(2));
        }
        
        payload += tlv('58', countryCode);
        payload += tlv('59', receiverName);
        payload += tlv('60', receiverCity);
        payload += tlv('62', tlv('05', '***')); 
        
        payload += '6304';
        
        const crc = this.crc16(payload);
        payload = payload.slice(0, -4) + '6304' + crc;
        
        return payload;
    },
    
    /**
     * Calcula CRC16 CCITT-FALSE
     */
    crc16(str) {
        let crc = 0xFFFF;
        for (let i = 0; i < str.length; i++) {
            crc ^= str.charCodeAt(i) << 8;
            for (let j = 0; j < 8; j++) {
                if (crc & 0x8000) {
                    crc = (crc << 1) ^ 0x1021;
                } else {
                    crc <<= 1;
                }
            }
        }
        return (crc & 0xFFFF).toString(16).toUpperCase().padStart(4, '0');
    },
    
    /**
     * Renderiza QR Code visual
     */
    renderQRCode(containerId, data) {
        const container = document.getElementById(containerId);
        if (!container) return;
        container.innerHTML = '';
        
        if (typeof qrcode === 'undefined') {
            container.innerHTML = '<p class="text-danger">Biblioteca QRCode não carregada</p>';
            return;
        }

        const qr = qrcode(0, 'M');
        qr.addData(data);
        qr.make();
        
        container.innerHTML = qr.createImgTag(4, 8);
    },
    
    /**
     * Gera QR Code de teste na página de config
     */
    generateTestQrCode() {
        const pixKey = this.getVal('pix-key');
        const receiverName = this.getVal('pix-receiver-name', 'PAGAMENTO');
        const receiverCity = this.getVal('pix-receiver-city', 'SAO PAULO');
        
        if (!pixKey) {
            Swal.fire('Erro', 'Configure a chave PIX primeiro.', 'error');
            return;
        }
        
        const pixCode = this.generateCode(pixKey, receiverName, receiverCity, 1.00, 'TESTE NEXUS');
        
        const container = document.getElementById('test-qrcode-container');
        if (container) {
            const qr = qrcode(0, 'M');
            qr.addData(pixCode);
            qr.make();
            container.innerHTML = qr.createImgTag(6, 0);
        }
        
        const codeEl = document.getElementById('test-pixcode');
        if (codeEl) codeEl.textContent = pixCode;
        
        const modal = new bootstrap.Modal(document.getElementById('modal-test-pix'));
        modal.show();
    },

    copyTestCode() {
        const code = document.getElementById('test-pixcode')?.textContent;
        if (!code) return;
        const el = document.createElement('textarea');
        el.value = code;
        document.body.appendChild(el);
        el.select();
        document.execCommand('copy');
        document.body.removeChild(el);
        Swal.fire({ title: 'Copiado!', icon: 'success', toast: true, position: 'top-end', showConfirmButton: false, timer: 2000 });
    },
    
    /**
     * Copia código PIX para clipboard
     */
    copyCode() {
        const code = document.getElementById('pix-test-code');
        if (!code) return;
        code.select();
        document.execCommand('copy');
        alert('Código PIX copiado!');
    },
    
    /**
     * Gera QR Code de teste no modal
     */
    generateModalTestQrCode() {
        const pixKey = this.getVal('edit-pix-key');
        const receiverName = this.getVal('edit-pix-receiver-name', 'PAGAMENTO');
        const receiverCity = this.getVal('edit-pix-receiver-city', 'SAO PAULO');
        const amount = parseFloat(this.getVal('modal-pix-test-value', '1.00')) || 1.00;
        const description = this.getVal('modal-pix-test-description');
        
        if (!pixKey) {
            alert('Configure a chave PIX primeiro');
            return;
        }
        
        const pixCode = this.generateCode(pixKey, receiverName, receiverCity, amount, description);
        
        this.renderQRCode('modal-pix-test-qrcode', pixCode);
        this.setVal('modal-pix-test-code', pixCode);
        
        const display = document.getElementById('modal-pix-test-value-display');
        if (display) display.textContent = typeof Format !== 'undefined' ? Format.brl(amount) : 'R$ ' + amount.toFixed(2);
        
        document.getElementById('modal-pix-test-result')?.classList.remove('hidden');
    },
    
    /**
     * Copia código PIX do modal
     */
    copyModalCode() {
        const code = document.getElementById('modal-pix-test-code');
        if (!code) return;
        code.select();
        document.execCommand('copy');
        alert('Código PIX copiado!');
    }
};

// Event listeners
document.addEventListener('DOMContentLoaded', () => {
    const pixSelector = document.getElementById('pix-site-selector');
    if (pixSelector) {
        pixSelector.addEventListener('change', (e) => PIX.onSiteChange(e.target.value));
    }
    
    const pixForm = document.getElementById('pix-config-form');
    if (pixForm) {
        pixForm.addEventListener('submit', (e) => PIX.saveConfig(e));
    }
});
