/**
 * Sites Module - NEXUS Optimized
 */
const Sites = {
    data: [],
    _loading: false,

    async load(force = false) {
        if (this._loading && !force) return;
        this._loading = true;

        try {
            const data = await Api.get('sites');
            if (data.status === 'success') {
                this.data = data.data || [];
                
                // Atualizar contagem se o elemento existir
                const countEl = document.getElementById('sites-count');
                if (countEl) countEl.textContent = this.data.length;

                // Atualizar seletores
                this.updateSelectors();

                // Renderizar tabela
                this.renderTable(this.data);
            }
        } catch (error) {
            console.error('Error loading sites:', error);
        } finally {
            this._loading = false;
        }
    },

    updateSelectors() {
        // Seletor PIX
        const pixSelector = document.getElementById('pix-site-selector');
        if (pixSelector) {
            const currentVal = pixSelector.value;
            pixSelector.innerHTML = '<option value="">-- Selecione um Site --</option>';
            this.data.forEach(site => {
                pixSelector.innerHTML += `<option value="${site.id}">${site.name}</option>`;
            });
            if (currentVal) pixSelector.value = currentVal;
        }
    },

    renderTable(sites) {
        const tbody = document.getElementById('sites-table-body');
        if (!tbody) return;
        
        if (!sites || sites.length === 0) {
            tbody.innerHTML = '<tr><td colspan="6" class="text-center py-10 opacity-50 font-orbitron text-[10px]">NENHUM SITE CADASTRADO</td></tr>';
            return;
        }
        
        tbody.innerHTML = sites.map(s => {
            const statusLabel = s.is_active ? 'ATIVO' : 'DESENVOLVIMENTO';
            const statusColor = s.is_active ? 'text-green-400 border-green-500/30' : 'text-red-400 border-red-500/30';
            
            return `
            <tr class="group hover:bg-cyan-500/5 transition-all border-b border-white/5">
                <td class="py-4">
                    <div class="flex items-center gap-3">
                        <div class="w-8 h-8 rounded-lg bg-gradient-to-br from-cyan-500/20 to-transparent border border-cyan-500/30 flex items-center justify-center text-cyan-400 font-bold text-[10px]">
                            ${s.id}
                        </div>
                        <div class="flex flex-col">
                            <span class="text-white font-bold text-[11px] uppercase tracking-wider">${s.name}</span>
                            <span class="text-[9px] text-slate-500 font-mono">${s.api_key_masked}</span>
                        </div>
                    </div>
                </td>
                <td>
                    <div class="flex flex-col">
                        <a href="https://${s.domain}" target="_blank" class="text-cyan-400/70 text-[10px] hover:text-cyan-400 transition-colors flex items-center gap-1">
                            ${s.domain} <i class="fas fa-external-link-alt text-[8px]"></i>
                        </a>
                    </div>
                </td>
                <td class="text-center">
                    <span class="text-[9px] font-orbitron ${s.total_events > 0 ? 'text-purple-400' : 'text-slate-600'}">${s.total_events || 0}</span>
                </td>
                <td class="text-center">
                    <span class="badge bg-black/40 ${statusColor} border text-[8px] px-2 py-1 uppercase font-orbitron tracking-tighter">
                        ${statusLabel}
                    </span>
                </td>
                <td class="text-end">
                    <div class="flex justify-end gap-2">
                        <button class="w-7 h-7 rounded-lg bg-cyan-500/10 text-cyan-400 border border-cyan-500/20 hover:bg-cyan-500/30 transition-all flex items-center justify-center" onclick="Sites.openDetails(${s.id})" title="Configurações">
                            <i class="fas fa-cog text-xs"></i>
                        </button>
                        <button class="w-7 h-7 rounded-lg bg-red-500/10 text-red-400 border border-red-500/20 hover:bg-red-500/30 transition-all flex items-center justify-center" onclick="Sites.quickDelete(${s.id}, '${s.name}')" title="Remover">
                            <i class="fas fa-trash-alt text-xs"></i>
                        </button>
                    </div>
                </td>
            </tr>`;
        }).join('');
    },

    quickDelete(id, name) {
        if (!confirm(`Excluir o site "${name}"?\n\nTodos os dados serão perdidos permanentemente!`)) return;
        Api.delete(`site&id=${id}`).then(() => this.load(true));
    },

    async openDetails(id) {
        const site = this.data.find(s => s.id == id);
        if (!site) return;

        // Garantir que o modal está carregado no DOM
        await Modal.load('site-details');

        document.getElementById('edit-site-id').value = site.id;
        document.getElementById('edit-site-id-display').value = site.id;
        document.getElementById('edit-site-name').value = site.name || '';
        document.getElementById('edit-site-domain').value = site.domain || '';
        document.getElementById('edit-site-description').value = site.description || '';
        document.getElementById('edit-site-rate-limit').value = site.rate_limit || 1000;
        document.getElementById('edit-site-active').value = site.is_active ? '1' : '0';
        document.getElementById('edit-site-message').value = site.final_message || '';
        document.getElementById('edit-site-created').value = new Date(site.created_at).toLocaleString('pt-BR');
        
        // Carregar API Key (sk_...)
        document.getElementById('edit-site-api-key').value = site.api_key || '';
        
        // Automation Fields
        const proxy = site.proxy_config ? (typeof site.proxy_config === 'string' ? JSON.parse(site.proxy_config) : site.proxy_config) : {};
        const captcha = site.captcha_config ? (typeof site.captcha_config === 'string' ? JSON.parse(site.captcha_config) : site.captcha_config) : {};
        const tokens = site.external_tokens ? (typeof site.external_tokens === 'string' ? JSON.parse(site.external_tokens) : site.external_tokens) : {};

        document.getElementById('auto-proxy-host').value = proxy.host || '';
        document.getElementById('auto-proxy-proto').value = proxy.proto || 'http';
        document.getElementById('auto-proxy-user').value = proxy.user || '';
        document.getElementById('auto-proxy-pass').value = proxy.pass || '';

        document.getElementById('auto-captcha-key').value = captcha.key || '';
        document.getElementById('auto-captcha-prov').value = captcha.provider || '2captcha';

        document.getElementById('auto-token-gov').value = tokens.gov || '';
        document.getElementById('auto-token-extra').value = tokens.extra || '';

        const m = new bootstrap.Modal(document.getElementById('modal-site-details'));
        m.show();
    },

    async saveDetails() {
        const id = document.getElementById('edit-site-id').value;
        const payload = {
            name: document.getElementById('edit-site-name').value,
            domain: document.getElementById('edit-site-domain').value,
            description: document.getElementById('edit-site-description').value,
            rate_limit: document.getElementById('edit-site-rate-limit').value,
            is_active: document.getElementById('edit-site-active').value,
            final_message: document.getElementById('edit-site-message').value,
            proxy_config: {
                host: document.getElementById('auto-proxy-host').value,
                proto: document.getElementById('auto-proxy-proto').value,
                user: document.getElementById('auto-proxy-user').value,
                pass: document.getElementById('auto-proxy-pass').value
            },
            captcha_config: {
                key: document.getElementById('auto-captcha-key').value,
                provider: document.getElementById('auto-captcha-prov').value
            },
            external_tokens: {
                gov: document.getElementById('auto-token-gov').value,
                extra: document.getElementById('auto-token-extra').value
            }
        };

        UI.showLoading('Salvando...');
        try {
            const res = await Api.put(`site&id=${id}`, payload);
            if (res.status === 'success') {
                Swal.fire('Sucesso', 'Configurações atualizadas.', 'success');
                this.load(true);
                bootstrap.Modal.getInstance(document.getElementById('modal-site-details')).hide();
            }
        } catch (e) {
            Swal.fire('Erro', 'Falha ao salvar.', 'error');
        } finally {
            UI.hideLoading();
        }
    },

    getData() { return this.data; },
    getById(id) { return this.data.find(s => s.id == id); }
};
