#!/bin/bash

# Build Script - Admin Server Frontend
# Concatena e minifica JS/CSS

set -e

SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
cd "$SCRIPT_DIR"

echo "=== Admin Server Build ==="

# Cores
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# Diretórios
JS_DIR="assets/js"
CSS_DIR="assets/css"
DIST_DIR="assets/dist"

# Criar diretório de distribuição
mkdir -p "$DIST_DIR"

# Timestamp para cache busting
TIMESTAMP=$(date +%s)

echo -e "${YELLOW}1. Concatenando JavaScript...${NC}"

# Ordem dos módulos (dependências primeiro)
JS_FILES=(
    "$JS_DIR/api.js"
    "$JS_DIR/auth.js"
    "$JS_DIR/dashboard.js"
    "$JS_DIR/sites.js"
    "$JS_DIR/users.js"
    "$JS_DIR/pix.js"
    "$JS_DIR/events.js"
    "$JS_DIR/security.js"
    "$JS_DIR/api-tokens.js"
    "$JS_DIR/app.js"
)

# Concatenar JS
cat "${JS_FILES[@]}" > "$DIST_DIR/admin.bundle.js"
echo -e "${GREEN}   -> admin.bundle.js criado${NC}"

# Verificar se terser está instalado
if command -v terser &> /dev/null; then
    echo -e "${YELLOW}2. Minificando JavaScript com terser...${NC}"
    terser "$DIST_DIR/admin.bundle.js" -o "$DIST_DIR/admin.min.js" --compress --mangle
    echo -e "${GREEN}   -> admin.min.js criado${NC}"
elif command -v uglifyjs &> /dev/null; then
    echo -e "${YELLOW}2. Minificando JavaScript com uglifyjs...${NC}"
    uglifyjs "$DIST_DIR/admin.bundle.js" -o "$DIST_DIR/admin.min.js" --compress --mangle
    echo -e "${GREEN}   -> admin.min.js criado${NC}"
else
    echo -e "${YELLOW}2. Terser/UglifyJS não encontrado, usando versão bundled...${NC}"
    cp "$DIST_DIR/admin.bundle.js" "$DIST_DIR/admin.min.js"
    echo -e "${GREEN}   -> admin.min.js criado (não minificado)${NC}"
fi

echo -e "${YELLOW}3. Processando CSS...${NC}"

# Copiar CSS
cp "$CSS_DIR/admin.css" "$DIST_DIR/admin.bundle.css"

# Verificar se cleancss está instalado
if command -v cleancss &> /dev/null; then
    cleancss -o "$DIST_DIR/admin.min.css" "$DIST_DIR/admin.bundle.css"
    echo -e "${GREEN}   -> admin.min.css criado${NC}"
elif command -v csso &> /dev/null; then
    csso "$DIST_DIR/admin.bundle.css" --output "$DIST_DIR/admin.min.css"
    echo -e "${GREEN}   -> admin.min.css criado${NC}"
else
    # Minificação básica sem ferramentas externas
    sed 's/\/\*.*\*\///g; s/  */ /g; s/\n//g' "$DIST_DIR/admin.bundle.css" > "$DIST_DIR/admin.min.css"
    echo -e "${GREEN}   -> admin.min.css criado (minificação básica)${NC}"
fi

# Criar versão de produção do index.html
echo -e "${YELLOW}4. Criando index.prod.html...${NC}"

sed -e 's|<script src="assets/js/api.js"></script>||g' \
    -e 's|<script src="assets/js/auth.js"></script>||g' \
    -e 's|<script src="assets/js/dashboard.js"></script>||g' \
    -e 's|<script src="assets/js/sites.js"></script>||g' \
    -e 's|<script src="assets/js/users.js"></script>||g' \
    -e 's|<script src="assets/js/pix.js"></script>||g' \
    -e 's|<script src="assets/js/events.js"></script>||g' \
    -e 's|<script src="assets/js/security.js"></script>||g' \
    -e 's|<script src="assets/js/api-tokens.js"></script>||g' \
    -e "s|<script src=\"assets/js/app.js\"></script>|<script src=\"assets/dist/admin.min.js?v=$TIMESTAMP\"></script>|g" \
    -e "s|assets/css/admin.css|assets/dist/admin.min.css?v=$TIMESTAMP|g" \
    index.html > index.prod.html

echo -e "${GREEN}   -> index.prod.html criado${NC}"

# Tamanhos
echo ""
echo "=== Tamanhos dos arquivos ==="
echo "JS Bundle: $(wc -c < "$DIST_DIR/admin.bundle.js" | xargs) bytes"
echo "JS Min:    $(wc -c < "$DIST_DIR/admin.min.js" | xargs) bytes"
echo "CSS Bundle: $(wc -c < "$DIST_DIR/admin.bundle.css" | xargs) bytes"
echo "CSS Min:    $(wc -c < "$DIST_DIR/admin.min.css" | xargs) bytes"
echo ""
echo -e "${GREEN}Build completo!${NC}"
echo "Para usar em produção, use index.prod.html ou atualize index.html para usar os arquivos minificados."
