-- ============================================
-- ADMIN TRACKING - SCHEMA MySQL COMPLETO
-- Sistema centralizado multi-site
-- ============================================

SET NAMES utf8mb4;
SET FOREIGN_KEY_CHECKS = 0;

-- ============================================
-- TABELA: admin_users (usuários do painel)
-- ============================================
DROP TABLE IF EXISTS admin_users;
CREATE TABLE admin_users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    email VARCHAR(100) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    role ENUM('super_admin', 'admin', 'viewer') DEFAULT 'admin',
    site_id INT NULL COMMENT 'NULL = super admin, ID = admin de site específico',
    is_active TINYINT(1) DEFAULT 1,
    last_login DATETIME NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_site (site_id),
    INDEX idx_role (role)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: sites (sites/clientes cadastrados)
-- ============================================
DROP TABLE IF EXISTS sites;
CREATE TABLE sites (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    domain VARCHAR(255) NOT NULL,
    api_key VARCHAR(64) NOT NULL UNIQUE,
    api_secret VARCHAR(128) NOT NULL,
    description TEXT NULL,
    is_active TINYINT(1) DEFAULT 1,
    rate_limit INT DEFAULT 1000 COMMENT 'Requisições por minuto',
    api_token_id INT NULL COMMENT 'Token de API externa vinculado',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_api_key (api_key),
    INDEX idx_domain (domain),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: admin_sessions (sessões do admin)
-- ============================================
DROP TABLE IF EXISTS admin_sessions;
CREATE TABLE admin_sessions (
    id VARCHAR(128) PRIMARY KEY,
    user_id INT NOT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    expires_at DATETIME NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_user (user_id),
    INDEX idx_expires (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: site_user_permissions (permissões por usuário/site)
-- ============================================
DROP TABLE IF EXISTS site_user_permissions;
CREATE TABLE site_user_permissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    site_id INT NULL,
    permission VARCHAR(50) NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_user_site_perm (user_id, site_id, permission),
    INDEX idx_user (user_id),
    INDEX idx_site (site_id),
    INDEX idx_permission (permission)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: api_tokens (tokens de APIs externas - DETRAN, etc)
-- ============================================
DROP TABLE IF EXISTS api_tokens;
CREATE TABLE api_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    token_value TEXT NOT NULL,
    base_url VARCHAR(500) NULL,
    description TEXT NULL,
    is_active TINYINT(1) DEFAULT 1,
    expires_at DATETIME NULL,
    created_by INT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_name (name),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: site_settings (configurações por site)
-- ============================================
DROP TABLE IF EXISTS site_settings;
CREATE TABLE site_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    setting_key VARCHAR(100) NOT NULL,
    setting_value TEXT NOT NULL,
    is_encrypted TINYINT(1) DEFAULT 0,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_site_setting (site_id, setting_key),
    INDEX idx_site_key (site_id, setting_key)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: global_settings (configurações globais)
-- ============================================
DROP TABLE IF EXISTS global_settings;
CREATE TABLE global_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) NOT NULL UNIQUE,
    setting_value TEXT NOT NULL,
    description VARCHAR(255) NULL,
    is_encrypted TINYINT(1) DEFAULT 0,
    updated_by INT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_key (setting_key)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: pix_config (configuração PIX por site)
-- ============================================
DROP TABLE IF EXISTS pix_config;
CREATE TABLE pix_config (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL UNIQUE,
    pix_key VARCHAR(255) NOT NULL COMMENT 'Criptografado',
    pix_key_type ENUM('cpf', 'cnpj', 'email', 'phone', 'random') DEFAULT 'random',
    receiver_name VARCHAR(25) NOT NULL DEFAULT 'PAGAMENTO',
    receiver_city VARCHAR(15) NOT NULL DEFAULT 'SAO PAULO',
    is_active TINYINT(1) DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_site (site_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: pix_config_history (histórico de alterações)
-- ============================================
DROP TABLE IF EXISTS pix_config_history;
CREATE TABLE pix_config_history (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    change_type ENUM('create', 'update', 'delete') DEFAULT 'update',
    old_pix_key VARCHAR(255) NULL,
    new_pix_key VARCHAR(255) NULL,
    old_receiver_name VARCHAR(25) NULL,
    new_receiver_name VARCHAR(25) NULL,
    old_receiver_city VARCHAR(15) NULL,
    new_receiver_city VARCHAR(15) NULL,
    changed_by_ip VARCHAR(45) NULL,
    changed_by_user_id INT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_site_date (site_id, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: events (eventos de tracking)
-- ============================================
DROP TABLE IF EXISTS events;
CREATE TABLE events (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    event_type VARCHAR(50) NOT NULL,
    session_id VARCHAR(64) NULL,
    visitor_id VARCHAR(64) NULL COMMENT 'ID único do visitante',
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    referrer VARCHAR(500) NULL,
    page_url VARCHAR(500) NULL,

    -- Dados do veículo/transação
    placa VARCHAR(10) NULL,
    renavam VARCHAR(20) NULL,
    valor DECIMAL(12,2) NULL,

    -- Geolocalização
    geo_city VARCHAR(100) NULL,
    geo_region VARCHAR(100) NULL,
    geo_country VARCHAR(10) NULL,
    geo_isp VARCHAR(200) NULL,

    -- UTM Parameters
    utm_source VARCHAR(100) NULL,
    utm_medium VARCHAR(100) NULL,
    utm_campaign VARCHAR(100) NULL,
    utm_term VARCHAR(100) NULL,
    utm_content VARCHAR(100) NULL,

    -- Metadados extras (JSON)
    metadata JSON NULL,

    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,

    INDEX idx_site_type (site_id, event_type),
    INDEX idx_site_date (site_id, created_at),
    INDEX idx_visitor (visitor_id),
    INDEX idx_session (session_id),
    INDEX idx_placa (placa),
    INDEX idx_created (created_at),
    INDEX idx_type_date (event_type, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: unique_visitors (visitantes únicos por dia)
-- ============================================
DROP TABLE IF EXISTS unique_visitors;
CREATE TABLE unique_visitors (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    visitor_id VARCHAR(64) NOT NULL,
    visit_date DATE NOT NULL,
    first_seen DATETIME NOT NULL,
    last_seen DATETIME NOT NULL,
    page_views INT DEFAULT 1,
    ip_address VARCHAR(45) NULL,
    geo_city VARCHAR(100) NULL,
    geo_region VARCHAR(100) NULL,
    user_agent TEXT NULL,
    UNIQUE KEY unique_visitor_day (site_id, visitor_id, visit_date),
    INDEX idx_site_date (site_id, visit_date),
    INDEX idx_visitor (visitor_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: daily_stats (estatísticas diárias pré-calculadas)
-- ============================================
DROP TABLE IF EXISTS daily_stats;
CREATE TABLE daily_stats (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    stat_date DATE NOT NULL,
    total_visits INT DEFAULT 0,
    unique_visitors INT DEFAULT 0,
    total_consultas INT DEFAULT 0,
    total_pix_gerados INT DEFAULT 0,
    total_pix_copiados INT DEFAULT 0,
    total_pix_pagos INT DEFAULT 0,
    valor_pix_gerado DECIMAL(12,2) DEFAULT 0,
    valor_pix_pago DECIMAL(12,2) DEFAULT 0,
    calculated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_site_date (site_id, stat_date),
    INDEX idx_site_date (site_id, stat_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: rate_limits (controle de rate limit)
-- ============================================
DROP TABLE IF EXISTS rate_limits;
CREATE TABLE rate_limits (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    ip_address VARCHAR(45) NOT NULL,
    request_count INT DEFAULT 1,
    window_start DATETIME NOT NULL,
    UNIQUE KEY unique_site_ip_window (site_id, ip_address, window_start),
    INDEX idx_cleanup (window_start)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: audit_logs (logs de auditoria)
-- ============================================
DROP TABLE IF EXISTS audit_logs;
CREATE TABLE audit_logs (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NULL,
    user_id INT NULL,
    action VARCHAR(100) NOT NULL,
    target_type VARCHAR(50) NULL COMMENT 'site, user, pix, etc',
    target_id INT NULL,
    details TEXT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_site (site_id),
    INDEX idx_user (user_id),
    INDEX idx_action (action),
    INDEX idx_target (target_type, target_id),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: site_security_config (configurações de segurança por site)
-- ============================================
DROP TABLE IF EXISTS site_security_config;
CREATE TABLE site_security_config (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL UNIQUE,

    -- Rate Limit
    rate_limit_enabled TINYINT(1) DEFAULT 0,
    rate_limit_requests INT DEFAULT 100,
    rate_limit_window INT DEFAULT 60 COMMENT 'segundos',
    rate_limit_block_duration INT DEFAULT 3600 COMMENT 'segundos',

    -- Anti-bot
    antibot_enabled TINYINT(1) DEFAULT 0,
    antibot_check_useragent TINYINT(1) DEFAULT 1,
    antibot_check_referrer TINYINT(1) DEFAULT 0,
    antibot_check_javascript TINYINT(1) DEFAULT 1,
    antibot_captcha_enabled TINYINT(1) DEFAULT 0,

    -- DDoS
    ddos_enabled TINYINT(1) DEFAULT 0,
    ddos_max_connections INT DEFAULT 50,
    ddos_suspicious_threshold INT DEFAULT 10,
    ddos_block_duration INT DEFAULT 7200 COMMENT 'segundos',

    -- Token GOV/DETRAN
    gov_token TEXT NULL COMMENT 'Criptografado',
    gov_token_expires_at DATETIME NULL,

    -- Bloqueios de interface
    block_devtools TINYINT(1) DEFAULT 0,
    block_right_click TINYINT(1) DEFAULT 0,
    block_view_source TINYINT(1) DEFAULT 0,
    block_copy TINYINT(1) DEFAULT 0,

    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_site (site_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: site_ip_rules (regras de IP por site)
-- ============================================
DROP TABLE IF EXISTS site_ip_rules;
CREATE TABLE site_ip_rules (
    id INT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    ip_address VARCHAR(45) NOT NULL,
    rule_type ENUM('whitelist', 'blacklist', 'temp_block') DEFAULT 'blacklist',
    reason VARCHAR(255) NULL,
    blocked_until DATETIME NULL COMMENT 'Para temp_block',
    created_by INT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    UNIQUE KEY unique_site_ip (site_id, ip_address),
    INDEX idx_site (site_id),
    INDEX idx_ip (ip_address),
    INDEX idx_type (rule_type)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- TABELA: security_logs (logs de segurança)
-- ============================================
DROP TABLE IF EXISTS security_logs;
CREATE TABLE security_logs (
    id BIGINT AUTO_INCREMENT PRIMARY KEY,
    site_id INT NOT NULL,
    event_type VARCHAR(50) NOT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    details JSON NULL,
    severity ENUM('low', 'medium', 'high', 'critical') DEFAULT 'low',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_site (site_id),
    INDEX idx_type (event_type),
    INDEX idx_severity (severity),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- FOREIGN KEYS (adicionar após criação das tabelas)
-- ============================================
ALTER TABLE admin_users
    ADD CONSTRAINT fk_users_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE SET NULL;

ALTER TABLE admin_sessions
    ADD CONSTRAINT fk_sessions_user FOREIGN KEY (user_id) REFERENCES admin_users(id) ON DELETE CASCADE;

ALTER TABLE site_user_permissions
    ADD CONSTRAINT fk_perms_user FOREIGN KEY (user_id) REFERENCES admin_users(id) ON DELETE CASCADE,
    ADD CONSTRAINT fk_perms_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE sites
    ADD CONSTRAINT fk_sites_api_token FOREIGN KEY (api_token_id) REFERENCES api_tokens(id) ON DELETE SET NULL;

ALTER TABLE site_settings
    ADD CONSTRAINT fk_settings_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE pix_config
    ADD CONSTRAINT fk_pix_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE pix_config_history
    ADD CONSTRAINT fk_pix_history_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE,
    ADD CONSTRAINT fk_pix_history_user FOREIGN KEY (changed_by_user_id) REFERENCES admin_users(id) ON DELETE SET NULL;

ALTER TABLE events
    ADD CONSTRAINT fk_events_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE unique_visitors
    ADD CONSTRAINT fk_visitors_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE daily_stats
    ADD CONSTRAINT fk_stats_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE rate_limits
    ADD CONSTRAINT fk_rates_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE site_security_config
    ADD CONSTRAINT fk_security_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE site_ip_rules
    ADD CONSTRAINT fk_ip_rules_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE,
    ADD CONSTRAINT fk_ip_rules_user FOREIGN KEY (created_by) REFERENCES admin_users(id) ON DELETE SET NULL;

ALTER TABLE security_logs
    ADD CONSTRAINT fk_sec_logs_site FOREIGN KEY (site_id) REFERENCES sites(id) ON DELETE CASCADE;

ALTER TABLE global_settings
    ADD CONSTRAINT fk_global_user FOREIGN KEY (updated_by) REFERENCES admin_users(id) ON DELETE SET NULL;

ALTER TABLE api_tokens
    ADD CONSTRAINT fk_api_tokens_user FOREIGN KEY (created_by) REFERENCES admin_users(id) ON DELETE SET NULL;

SET FOREIGN_KEY_CHECKS = 1;

-- ============================================
-- INSERIR USUÁRIO ADMIN PADRÃO
-- Senha: admin123 (MUDE APÓS INSTALAÇÃO!)
-- ============================================
INSERT INTO admin_users (username, email, password_hash, role, site_id) VALUES
('admin', 'admin@admin.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'super_admin', NULL)
ON DUPLICATE KEY UPDATE username = username;

-- ============================================
-- PROCEDURES
-- ============================================

DELIMITER //

DROP PROCEDURE IF EXISTS calculate_daily_stats//
CREATE PROCEDURE calculate_daily_stats(IN p_site_id INT, IN p_date DATE)
BEGIN
    INSERT INTO daily_stats (site_id, stat_date, total_visits, unique_visitors, total_consultas,
                            total_pix_gerados, total_pix_copiados, total_pix_pagos,
                            valor_pix_gerado, valor_pix_pago)
    SELECT
        p_site_id,
        p_date,
        COUNT(CASE WHEN event_type IN ('visit', 'page_view', 'page_load') THEN 1 END),
        (SELECT COUNT(DISTINCT visitor_id) FROM events WHERE site_id = p_site_id AND DATE(created_at) = p_date),
        COUNT(CASE WHEN event_type IN ('cnpj_consulta', 'consulta', 'consulta_iniciada', 'consulta_placa') THEN 1 END),
        COUNT(CASE WHEN event_type IN ('pix_gerado', 'pix_generation_success', 'pix_generated', 'pix_iniciado') THEN 1 END),
        COUNT(CASE WHEN event_type IN ('pix_codigo_copiado', 'pix_copied', 'pix_copia_cola') THEN 1 END),
        COUNT(CASE WHEN event_type IN ('pix_pago', 'pix_paid') THEN 1 END),
        COALESCE(SUM(CASE WHEN event_type IN ('pix_gerado', 'pix_generation_success', 'pix_generated', 'pix_iniciado') THEN valor ELSE 0 END), 0),
        COALESCE(SUM(CASE WHEN event_type IN ('pix_pago', 'pix_paid') THEN valor ELSE 0 END), 0)
    FROM events
    WHERE site_id = p_site_id AND DATE(created_at) = p_date
    ON DUPLICATE KEY UPDATE
        total_visits = VALUES(total_visits),
        unique_visitors = VALUES(unique_visitors),
        total_consultas = VALUES(total_consultas),
        total_pix_gerados = VALUES(total_pix_gerados),
        total_pix_copiados = VALUES(total_pix_copiados),
        total_pix_pagos = VALUES(total_pix_pagos),
        valor_pix_gerado = VALUES(valor_pix_gerado),
        valor_pix_pago = VALUES(valor_pix_pago);
END //

DROP PROCEDURE IF EXISTS cleanup_rate_limits//
CREATE PROCEDURE cleanup_rate_limits()
BEGIN
    DELETE FROM rate_limits WHERE window_start < DATE_SUB(NOW(), INTERVAL 1 HOUR);
END //

DROP PROCEDURE IF EXISTS cleanup_sessions//
CREATE PROCEDURE cleanup_sessions()
BEGIN
    DELETE FROM admin_sessions WHERE expires_at < NOW();
END //

DROP PROCEDURE IF EXISTS cleanup_temp_blocks//
CREATE PROCEDURE cleanup_temp_blocks()
BEGIN
    DELETE FROM site_ip_rules WHERE rule_type = 'temp_block' AND blocked_until < NOW();
END //

DELIMITER ;

-- ============================================
-- FIM DO SCHEMA
-- ============================================
