<?php
/**
 * Dashboard NEXUS v4.0 - Full Responsive + Export
 */
session_start();
require_once __DIR__ . '/config/database.php';

spl_autoload_register(function ($class) {
    $nsMap = ['Core\\' => __DIR__ . '/api/core/', 'Middleware\\' => __DIR__ . '/api/middleware/', 'Controllers\\' => __DIR__ . '/api/controllers/'];
    foreach ($nsMap as $ns => $dir) {
        if (strpos($class, $ns) === 0) {
            $rel = substr($class, strlen($ns));
            $file = $dir . str_replace('\\', '/', $rel) . '.php';
            if (file_exists($file)) { require_once $file; return true; }
        }
    }
    return false;
});

use Middleware\Auth;
$user = Auth::getUser();
if (!$user) { header('Location: index.php'); exit; }
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Nexus - Central de Inteligência</title>
    <link rel="icon" type="image/png" href="/logo.png">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link href="https://fonts.googleapis.com/css2?family=Orbitron:wght@400;700&family=Inter:wght@300;400;600&display=swap" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.tailwindcss.com"></script>
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script src="https://cdn.jsdelivr.net/npm/axios/dist/axios.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/qrcode-generator@1.4.4/qrcode.min.js"></script>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/themes/dark.css">
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
    <style>
        :root { --accent-cyan: #00e5ff; --bg-dark: #05070a; --card-bg: rgba(10, 16, 26, 0.95); --border-color: rgba(0, 229, 255, 0.2); }
        body { background-color: var(--bg-dark); color: #fff; font-family: "Inter", sans-serif; min-height: 100vh; margin: 0; overflow-x: hidden; }
        .font-orbitron { font-family: "Orbitron", sans-serif; }
        
        /* Sidebar Responsive Logic */
        .sidebar { width: 260px; background: rgba(10, 16, 26, 0.98); border-right: 1px solid var(--border-color); position: fixed; height: 100%; z-index: 2000; transition: transform 0.3s ease; }
        .main { margin-left: 260px; padding: 20px; transition: margin-left 0.3s ease; }
        
        @media (max-width: 992px) {
            .sidebar { transform: translateX(-100%); }
            .sidebar.open { transform: translateX(0); }
            .main { margin-left: 0; }
            .topbar { flex-wrap: wrap; gap: 15px; }
        }

        .menu-link { display: flex; align-items: center; gap: 12px; padding: 12px 20px; color: #94a3b8; border-radius: 12px; margin: 4px 15px; transition: 0.3s; text-decoration: none !important; cursor: pointer; border-left: 3px solid transparent; }
        .menu-link:hover, .menu-link.active { background: rgba(0, 229, 255, 0.1); color: var(--accent-cyan); border-left: 3px solid var(--accent-cyan); }
        
        .topbar { background: var(--card-bg); border: 1px solid var(--border-color); border-radius: 15px; padding: 12px 20px; margin-bottom: 25px; display: flex; justify-content: space-between; align-items: center; backdrop-filter: blur(10px); }
        .nexus-card { background: var(--card-bg); border: 1px solid var(--border-color); border-radius: 20px; padding: 20px; box-shadow: 0 10px 30px rgba(0,0,0,0.5); }
        
        .stat-card { 
            position: relative;
            padding: 0; 
            border-radius: 18px; 
            border: 1px solid var(--border-color); 
            background: rgba(10,16,26,0.6); 
            overflow: hidden;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            cursor: pointer;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            border-color: var(--accent-cyan);
            box-shadow: 0 0 20px rgba(0, 229, 255, 0.2);
            background: rgba(0, 229, 255, 0.05);
        }

        .stat-card-header {
            padding: 10px;
            background: rgba(0, 229, 255, 0.03);
            border-bottom: 1px solid rgba(0, 229, 255, 0.1);
            font-size: 9px;
            font-family: 'Orbitron';
            text-transform: uppercase;
            color: #64748b;
        }

        .stat-card-body {
            display: flex;
            align-items: center;
            padding: 12px;
        }

        .stat-value-container {
            flex: 1;
            text-align: center;
        }

        .stat-value-label {
            font-size: 8px;
            color: #475569;
            text-transform: uppercase;
            margin-bottom: 2px;
        }

        .stat-value-number {
            font-family: 'Orbitron';
            font-weight: 700;
            font-size: 16px;
            line-height: 1;
        }

        .stat-divider {
            width: 1px;
            height: 30px;
            background: rgba(0, 229, 255, 0.1);
            margin: 0 5px;
        }
        
        .nav-tabs { border: none !important; margin-bottom: 20px; display: flex; overflow-x: auto; flex-wrap: nowrap; -webkit-overflow-scrolling: touch; }
        .nav-link { white-space: nowrap; color: #64748b; font-family: 'Orbitron'; font-size: 10px; border: none !important; }
        .nav-link.active { color: var(--accent-cyan) !important; background: rgba(0, 229, 255, 0.1) !important; border-radius: 8px; }

        .nexus-table { width: 100%; color: #fff; }
        .nexus-table th { padding: 12px; color: var(--accent-cyan); font-family: 'Orbitron'; font-size: 9px; text-transform: uppercase; border-bottom: 1px solid var(--border-color); }
        .nexus-table td { padding: 12px; background: rgba(255,255,255,0.02); font-size: 12px; border-bottom: 1px solid rgba(255,255,255,0.05); }

        #space-bg { position: fixed; top: 0; left: 0; width: 100%; height: 100%; z-index: -1; background: radial-gradient(circle at 50% 50%, #0a192f 0%, #05070a 100%); }
        .star { position: absolute; background: white; border-radius: 50%; opacity: 0.3; }
        
        .btn-nexus { padding: 8px 16px; border-radius: 10px; font-family: 'Orbitron'; font-size: 10px; font-weight: bold; transition: 0.3s; display: inline-flex; align-items: center; gap: 8px; }
        .btn-clear { border: 1px solid #ef4444; color: #ef4444; background: #ef444410; }
        .btn-clear:hover { background: #ef4444; color: #fff; }
        .btn-export { border: 1px solid #10b981; color: #10b981; background: #10b98110; }
        .btn-export:hover { background: #10b981; color: #fff; }
        
        .page-content { display: none; }
        .page-content.active { display: block; }
        .mobile-toggle { display: none; font-size: 24px; color: var(--accent-cyan); cursor: pointer; }
        @media (max-width: 992px) { .mobile-toggle { display: block; } }

        .has-security-event { background: rgba(0, 229, 255, 0.2) !important; border: 1px solid var(--accent-cyan) !important; color: #fff !important; }
    </style>
</head>
<body>
    <div id="space-bg"></div>
    
    <!-- SIDEBAR -->
    <aside class="sidebar" id="sidebar">
        <div class="p-6 text-center border-b border-white/5">
            <img src="/logo.png" class="h-12 mx-auto mb-3">
            <div class="font-orbitron text-[9px] text-cyan-400 font-bold uppercase tracking-tighter">Central Nexus</div>
        </div>
        <div class="mt-4">
            <a onclick="App.showPage('dashboard')" class="menu-link active" data-page="dashboard"><i class="fas fa-chart-line"></i> Dashboard</a>
            <a onclick="App.showPage('automation')" class="menu-link" data-page="automation"><i class="fas fa-robot"></i> Automação</a>
            <a onclick="App.showPage('analytics')" class="menu-link" data-page="analytics"><i class="fas fa-brain"></i> Inteligência</a>
            <a onclick="App.showPage('finance')" class="menu-link" data-page="finance"><i class="fas fa-hand-holding-usd"></i> Financeiro</a>
            <a onclick="App.showPage('sites')" class="menu-link" data-page="sites"><i class="fas fa-globe"></i> Sites</a>
            <a onclick="App.showPage('pix')" class="menu-link" data-page="pix"><i class="fas fa-qrcode"></i> Config PIX</a>
            <a onclick="App.showPage('settings')" class="menu-link" data-page="settings"><i class="fas fa-cog"></i> Configuração</a>
            <a onclick="App.showPage('users')" class="menu-link" data-page="users"><i class="fas fa-users-cog"></i> Usuários</a>
            <div class="border-t border-white/5 my-4 mx-4"></div>
            <a href="logout.php" class="menu-link text-red-400"><i class="fas fa-sign-out-alt"></i> Sair</a>
        </div>
    </aside>

    <main class="main">
        <header class="topbar">
            <div class="flex items-center gap-3">
                <div class="mobile-toggle" onclick="App.toggleSidebar()"><i class="fas fa-bars"></i></div>
                <div class="font-orbitron text-[9px] text-slate-400 uppercase hidden sm:block">Período: 
                    <select id="range-selector" class="bg-transparent text-cyan-400 border-none outline-none cursor-pointer text-[10px]" onchange="TabManager.load()">
                        <option value="today" class="bg-slate-900">Hoje</option>
                        <option value="yesterday" class="bg-slate-900">Ontem</option>
                        <option value="7days" class="bg-slate-900">7 Dias</option>
                        <option value="all" class="bg-slate-900">Tudo</option>
                    </select>
                </div>
            </div>
            <div class="flex items-center gap-3">
                <button class="btn-nexus btn-clear" onclick="TabManager.clearAll()"><i class="fas fa-trash"></i> <span class="hidden md:inline">LIMPAR</span></button>
                <button class="btn-nexus btn-export" onclick="TabManager.export()"><i class="fas fa-file-export"></i> <span class="hidden md:inline">EXPORTAR</span></button>
                <div class="w-px h-6 bg-white/10 mx-2"></div>
                <img src="https://ui-avatars.com/api/?name=<?php echo $user['username']; ?>&background=00e5ff&color=000" class="w-8 h-8 rounded-lg border border-cyan-500/30">
            </div>
        </header>

        <!-- PÁGINAS -->
        <div id="page-dashboard" class="page-content active">
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                <!-- VISITAS -->
                <div class="stat-card" onclick="TabManager.switch('visits')">
                    <div class="stat-card-header">Visitas</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Hoje</div>
                            <div class="stat-value-number" id="stat-visits-today">0</div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container">
                            <div class="stat-value-label">Total</div>
                            <div class="stat-value-number opacity-50" id="stat-visits-total">0</div>
                        </div>
                    </div>
                </div>

                <!-- CONSULTAS -->
                <div class="stat-card" onclick="TabManager.switch('consultas')">
                    <div class="stat-card-header text-blue-400">Consultas</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Hoje</div>
                            <div class="stat-value-number text-blue-400" id="stat-consultas-today">0</div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container">
                            <div class="stat-value-label">Total</div>
                            <div class="stat-value-number text-blue-400 opacity-50" id="stat-consultas-total">0</div>
                        </div>
                    </div>
                </div>

                <!-- GERADOS -->
                <div class="stat-card" onclick="TabManager.switch('pix_gen')">
                    <div class="stat-card-header text-purple-400">Gerados</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Hoje</div>
                            <div class="stat-value-number text-purple-400" id="stat-pix-today">0</div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container">
                            <div class="stat-value-label">Total</div>
                            <div class="stat-value-number text-purple-400 opacity-50" id="stat-pix-total">0</div>
                        </div>
                    </div>
                </div>

                <!-- PAGOS -->
                <div class="stat-card" onclick="TabManager.switch('pix_confirm')">
                    <div class="stat-card-header text-green-400">Pagos</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Hoje</div>
                            <div class="stat-value-number text-green-400" id="stat-pix-value-today">R$ 0</div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container">
                            <div class="stat-value-label">Total</div>
                            <div class="stat-value-number text-green-400 opacity-50" id="stat-pix-value-total">R$ 0</div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="nexus-card">
                <div class="flex flex-col md:flex-row gap-3 mb-4">
                    <div class="flex-1 relative">
                        <i class="fas fa-search absolute left-3 top-1/2 -translate-y-1/2 text-slate-500 text-xs"></i>
                        <input type="text" id="search-input" class="w-full bg-black/40 border border-white/10 rounded-xl py-2 pl-9 pr-4 text-xs focus:border-cyan-500/50 outline-none transition-all" placeholder="Pesquisar por Placa, IP ou WhatsApp..." onkeyup="if(event.key==='Enter') TabManager.load()">
                    </div>
                    <div class="flex gap-2">
                        <input type="date" id="date-start" class="bg-black/40 border border-white/10 rounded-xl py-2 px-3 text-[10px] outline-none" onchange="TabManager.load()">
                        <input type="date" id="date-end" class="bg-black/40 border border-white/10 rounded-xl py-2 px-3 text-[10px] outline-none" onchange="TabManager.load()">
                        <button class="btn-nexus bg-cyan-500/10 text-cyan-400 border border-cyan-500/20" onclick="TabManager.load()"><i class="fas fa-filter"></i></button>
                    </div>
                </div>

                <ul class="nav nav-tabs">
                    <li><button class="nav-link active" onclick="TabManager.switch('visits', this)">Visitas</button></li>
                    <li><button class="nav-link" onclick="TabManager.switch('consultas', this)">Consultas</button></li>
                    <li><button class="nav-link" onclick="TabManager.switch('pix_gen', this)">PIX Gerados</button></li>
                    <li><button class="nav-link" onclick="TabManager.switch('pix_confirm', this)">Pagos</button></li>
                    <li><button class="nav-link" onclick="TabManager.switch('blocked', this)">Bloqueados</button></li>
                </ul>
                <div class="table-responsive"><table class="nexus-table"><thead id="table-head"></thead><tbody id="table-body"></tbody></table></div>
            </div>
        </div>

        <!-- CENTRO DE AUTOMAÇÃO UNIFICADO -->
        <div id="page-automation" class="page-content">
            <div class="flex justify-between items-center mb-6">
                <h2 class="font-orbitron text-cyan-400 text-lg uppercase tracking-wider">Centro de Automação e Infraestrutura</h2>
                <div class="flex gap-2">
                    <button class="btn btn-nexus bg-purple-500/10 text-purple-400 border border-purple-500/20" onclick="Settings.testCaptcha()">
                        <i class="fas fa-puzzle-piece"></i> TESTAR CAPTCHA
                    </button>
                    <button class="btn btn-nexus bg-cyan-500/10 text-cyan-400 border border-cyan-500/20" onclick="Settings.testProxy()">
                        <i class="fas fa-network-wired"></i> TESTAR PROXY
                    </button>
                </div>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-8">
                <!-- SAÚDE DAS APIS -->
                <div class="stat-card border-cyan-500/20">
                    <div class="stat-card-header text-cyan-400">⚡ Status Detrans</div>
                    <div class="stat-card-body grid grid-cols-2 gap-2">
                        <div class="text-center p-2 bg-white/5 rounded-lg border border-white/5">
                            <div class="text-[8px] opacity-50 uppercase mb-1">Zapay</div>
                            <div class="text-[10px] font-bold" id="status-api-zapay">---</div>
                        </div>
                        <div class="text-center p-2 bg-white/5 rounded-lg border border-white/5">
                            <div class="text-[8px] opacity-50 uppercase mb-1">MS Placa</div>
                            <div class="text-[10px] font-bold" id="status-api-ms">---</div>
                        </div>
                        <div class="text-center p-2 bg-white/5 rounded-lg border border-white/5">
                            <div class="text-[8px] opacity-50 uppercase mb-1">MG Placa</div>
                            <div class="text-[10px] font-bold" id="status-api-mg">---</div>
                        </div>
                        <div class="text-center p-2 bg-white/5 rounded-lg border border-white/5">
                            <div class="text-[8px] opacity-50 uppercase mb-1">RJ Placa</div>
                            <div class="text-[10px] font-bold" id="status-api-rj">---</div>
                        </div>
                    </div>
                </div>

                <!-- CAPTCHA -->
                <div class="stat-card border-purple-500/20">
                    <div class="stat-card-header text-purple-400">🧩 Captcha (CapSolver)</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Hoje / Total</div>
                            <div class="stat-value-number text-purple-400"><span id="stat-solved-captcha">0</span> / <span class="text-[10px] opacity-50" id="stat-total-solved-captcha">0</span></div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container text-center cursor-pointer hover:bg-white/5 transition-colors p-1 rounded" onclick="View.showCaptchaLogs()">
                            <div class="stat-value-label">Saldo em Conta</div>
                            <div class="text-[11px] font-bold text-white mt-1" id="status-captcha">$ 0.0000</div>
                        </div>
                    </div>
                </div>

                <!-- PROXY E ROTATIVIDADE -->
                <div class="stat-card border-blue-500/20 cursor-pointer hover:bg-blue-500/5 transition-all" onclick="View.showProxyList()">
                    <div class="stat-card-header text-blue-400">🌐 Rede Proxy (Nodemaven)</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Hoje / Total</div>
                            <div class="stat-value-number text-blue-400"><span id="stat-proxy-requests">0</span> / <span class="text-[10px] opacity-50" id="stat-total-proxy-requests">0</span></div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container text-center">
                            <div class="stat-value-label">Sessões (Hoje)</div>
                            <div class="stat-value-number text-blue-400 opacity-50" id="stat-proxy-period">0</div>
                            <div class="text-[7px] opacity-50 uppercase mt-1">VER LISTA</div>
                        </div>
                    </div>
                </div>

                <!-- DEVTOOLS E INFRA -->
                <div class="stat-card border-yellow-500/20">
                    <div class="stat-card-header text-yellow-400">🛠️ Infra / DevTools</div>
                    <div class="stat-card-body">
                        <div class="stat-value-container">
                            <div class="stat-value-label">Interações de Proxy</div>
                            <div class="stat-value-number text-yellow-400" id="stat-devtools">0</div>
                        </div>
                        <div class="stat-divider"></div>
                        <div class="stat-value-container text-center">
                            <div class="stat-value-label">Status Server</div>
                            <div class="text-[10px] text-green-400 font-bold mt-1">ONLINE</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- SEGURANÇA E BLOQUEIOS -->
            <div class="nexus-card p-6">
                <div class="flex justify-between items-center mb-4 border-b border-white/5 pb-2">
                    <h6 class="font-orbitron text-xs text-red-400 m-0 uppercase tracking-tighter"><i class="fas fa-user-shield me-2"></i>Histórico de Bloqueios e Segurança</h6>
                    <div class="flex items-center gap-2">
                        <label class="text-[9px] opacity-50 uppercase font-orbitron">Filtrar Data:</label>
                        <input type="date" id="security-calendar" class="bg-black/40 border border-white/10 rounded px-2 py-1 text-[10px] text-cyan-400 outline-none w-32 cursor-pointer" onchange="Automation.loadLogs(1)">
                    </div>
                </div>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-2">
                    <div class="bg-black/20 p-3 rounded border border-white/5 text-center">
                        <div class="text-[10px] opacity-50 uppercase">BOTS BLOQUEADOS</div>
                        <div class="flex justify-center items-baseline gap-2">
                            <span class="text-lg font-bold text-red-400" id="stat-bots">0</span>
                            <span class="text-[9px] opacity-30" id="stat-bots-total">TOTAL: 0</span>
                        </div>
                    </div>
                    <div class="bg-black/20 p-3 rounded border border-white/5 text-center">
                        <div class="text-[10px] opacity-50 uppercase">REDIRECIONADOS</div>
                        <div class="flex justify-center items-baseline gap-2">
                            <span class="text-lg font-bold text-orange-400" id="stat-redirects">0</span>
                            <span class="text-[9px] opacity-30" id="stat-redirects-total">TOTAL: 0</span>
                        </div>
                    </div>
                    <div class="bg-black/20 p-3 rounded border border-white/5 text-center">
                        <div class="text-[10px] opacity-50 uppercase">ATAQUES DDOS</div>
                        <div class="flex justify-center items-baseline gap-2">
                            <span class="text-lg font-bold text-red-600" id="stat-ddos">0</span>
                            <span class="text-[9px] opacity-30" id="stat-ddos-total">TOTAL: 0</span>
                        </div>
                    </div>
                </div>
                <div class="mt-4 p-2 bg-red-500/5 rounded border border-red-500/10 flex items-center gap-3">
                    <i class="fas fa-calendar-check text-red-400 text-xs"></i>
                    <span class="text-[9px] text-slate-400 italic">O calendário acima destaca em <b class="text-cyan-400">ciano</b> os dias que tiveram interações de segurança detectadas.</span>
                </div>

                <!-- TABELA DE LOGS DE AUTOMAÇÃO -->
                <div class="mt-6">
                    <div class="flex justify-between items-center mb-3">
                        <h6 class="font-orbitron text-[10px] text-cyan-400 m-0 uppercase tracking-widest">Logs de Automação Recentes</h6>
                        <button class="text-[9px] bg-cyan-500/10 text-cyan-400 px-3 py-1 rounded border border-cyan-500/20 hover:bg-cyan-500/20 transition-all" onclick="Automation.loadLogs()">
                            <i class="fas fa-sync-alt me-1"></i> ATUALIZAR
                        </button>
                    </div>
                    <div class="table-responsive bg-black/20 rounded-xl border border-white/5">
                        <table class="nexus-table text-[10px]">
                            <thead>
                                <tr>
                                    <th>Site</th>
                                    <th>IP</th>
                                    <th>Evento</th>
                                    <th>Local</th>
                                    <th>Data/Hora</th>
                                </tr>
                            </thead>
                            <tbody id="automation-logs-body">
                                <tr><td colspan="5" class="text-center py-10 opacity-30">Carregando eventos...</td></tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <div id="page-sites" class="page-content">
            <div class="nexus-card">
                <div class="flex justify-between items-center mb-4"><h6 class="font-orbitron text-cyan-400">Sites</h6><button class="btn btn-primary btn-sm text-[10px]" data-bs-toggle="modal" data-bs-target="#modal-new-site">NOVO</button></div>
                <div class="table-responsive"><table class="table text-white"><thead><tr><th>Nome</th><th>Status</th><th>Ações</th></tr></thead><tbody id="sites-table-body"></tbody></table></div>
            </div>
        </div>

        <div id="page-pix" class="page-content">
            <div class="nexus-card">
                <h6 class="font-orbitron text-cyan-400 mb-4">Configuração PIX</h6>
                <select id="pix-site-selector" class="form-select form-select-sm bg-dark text-white mb-4"></select>
                <form id="pix-config-form" class="hidden">
                    <div class="row g-3">
                                                <div class="col-6">
                                                    <label class="text-[10px] uppercase opacity-50">Estado</label>
                                                    <select id="pix-state" class="form-select bg-dark text-white border-white/10 text-xs" onchange="PIX.loadConfig(PIX.currentSiteId)">
                                                        <option value="BR">Global (Todos)</option>
                                                        <option value="AC">Acre</option>
                        <option value="AL">Alagoas</option><option value="AP">Amapá</option>
                                <option value="AM">Amazonas</option><option value="BA">Bahia</option><option value="CE">Ceará</option>
                                <option value="DF">Distrito Federal</option><option value="ES">Espírito Santo</option><option value="GO">Goiás</option>
                                <option value="MA">Maranhão</option><option value="MT">Mato Grosso</option><option value="MS">Mato Grosso do Sul</option>
                                <option value="MG">Minas Gerais</option><option value="PA">Pará</option><option value="PB">Paraíba</option>
                                <option value="PR">Paraná</option><option value="PE">Pernambuco</option><option value="PI">Piauí</option>
                                <option value="RJ">Rio de Janeiro</option><option value="RN">Rio Grande do Norte</option><option value="RS">Rio Grande do Sul</option>
                                <option value="RO">Rondônia</option><option value="RR">Roraima</option><option value="SC">Santa Catarina</option>
                                <option value="SP">São Paulo</option><option value="SE">Sergipe</option><option value="TO">Tocantins</option>
                            </select>
                        </div>
                        <div class="col-6"><label class="text-[10px] uppercase opacity-50">Nome do Recebedor</label><input type="text" id="pix-receiver-name" class="form-control bg-dark text-white border-white/10 text-xs" placeholder="Ex: PAGAMENTO"></div>
                        <div class="col-6">
                            <label class="text-[10px] uppercase opacity-50">Tipo de Chave</label>
                            <select id="pix-key-type" class="form-select bg-dark text-white border-white/10 text-xs">
                                <option value="random">Chave Aleatória (EVP)</option>
                                <option value="cpf">CPF</option>
                                <option value="cnpj">CNPJ</option>
                                <option value="email">E-mail</option>
                                <option value="phone">Telefone (Celular)</option>
                            </select>
                        </div>
                        <div class="col-12"><label class="text-[10px] uppercase opacity-50">Chave PIX</label><input type="text" id="pix-key" class="form-control bg-dark text-white border-white/10 font-mono text-xs" placeholder="Cole sua chave aqui..."></div>
                        <div class="col-12"><label class="text-[10px] uppercase opacity-50">Cidade</label><input type="text" id="pix-receiver-city" class="form-control bg-dark text-white border-white/10 text-xs" value="SAO PAULO"></div>
                        <div class="col-12 flex gap-2">
                            <button type="submit" class="btn btn-primary flex-1 mt-3">Salvar Configuração</button>
                            <button type="button" class="btn btn-outline-warning mt-3 px-4" onclick="PIX.generateTestQrCode()">
                                <i class="fas fa-qrcode"></i> GERAR TESTE R$ 1,00
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Modal de Teste PIX -->
            <div class="modal fade" id="modal-test-pix" tabindex="-1" aria-hidden="true">
                <div class="modal-dialog modal-dialog-centered">
                    <div class="modal-content nexus-card">
                        <div class="modal-header border-0">
                            <h6 class="modal-title font-orbitron text-cyan-400">QR Code de Teste - R$ 1,00</h6>
                            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body text-center p-10">
                            <div id="test-qrcode-container" class="flex justify-center mb-6"></div>
                            <div class="bg-black/40 p-4 rounded-xl border border-white/5 font-mono text-[10px] break-all mb-4" id="test-pixcode"></div>
                            <button class="btn btn-sm btn-outline-cyan w-100" onclick="PIX.copyTestCode()">COPIAR CÓDIGO</button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Listagem de Chaves PIX -->
            <div class="nexus-card mt-6">
                <h6 class="font-orbitron text-cyan-400 mb-4 flex items-center gap-2">
                    <i class="fas fa-list"></i> Chaves Configuradas
                </h6>
                <div class="table-responsive">
                    <table class="nexus-table">
                        <thead>
                            <tr>
                                <th>Site</th>
                                <th>Estado</th>
                                <th>Tipo</th>
                                <th>Chave PIX</th>
                                <th class="text-end">Ações</th>
                            </tr>
                        </thead>
                        <tbody id="pix-configs-table-body">
                            <tr><td colspan="5" class="text-center py-4 opacity-50">Carregando chaves...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <div id="page-users" class="page-content">
            <div class="nexus-card">
                <h6 class="font-orbitron text-cyan-400 mb-4">Usuários</h6>
                <div class="table-responsive"><table class="table text-white"><tbody id="users-table-body"></tbody></table></div>
            </div>
        </div>

        <div id="page-analytics" class="page-content">
            <div class="row g-4 mb-6">
                <!-- Coluna de IA -->
                <div class="col-lg-4">
                    <div class="nexus-card h-full bg-gradient-to-b from-cyan-500/5 to-transparent border-cyan-500/20">
                        <div class="flex items-center gap-2 mb-4 text-cyan-400 font-orbitron text-xs">
                            <i class="fas fa-robot animate-pulse"></i> NEXUS AI INSIGHTS
                        </div>
                        <div id="ai-insights-container" class="space-y-3">
                            <!-- Injetado via JS -->
                        </div>
                        
                        <div class="mt-6 p-4 bg-black/40 rounded-2xl border border-white/5">
                            <div class="text-[9px] text-slate-500 uppercase font-orbitron mb-1">Previsão de Fechamento</div>
                            <div class="text-2xl font-bold font-orbitron text-white" id="ai-forecast">R$ 0,00</div>
                            <div class="text-[8px] text-cyan-500 mt-1">Baseado na tendência atual de tráfego</div>
                        </div>
                    </div>
                </div>

                <!-- Gráfico de Tendência -->
                <div class="col-lg-8">
                    <div class="nexus-card">
                        <div class="text-[10px] text-slate-500 font-orbitron uppercase mb-4">Evolução de Conversão (7 dias)</div>
                        <canvas id="conversionChart" height="150"></canvas>
                    </div>
                </div>
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <div class="nexus-card">
                    <div class="text-[10px] text-slate-500 font-orbitron uppercase mb-4 text-center">Funil de Interação (Hoje)</div>
                    <canvas id="funnelChart" height="200"></canvas>
                </div>
                <div class="nexus-card">
                    <div class="text-[10px] text-slate-500 font-orbitron uppercase mb-4">Ranking de Performance (Sites)</div>
                    <div id="site-ranking-body" class="space-y-4">
                        <!-- Injetado via JS -->
                    </div>
                </div>
            </div>
        </div>

        <div id="page-finance" class="page-content">
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                <div class="nexus-card bg-gradient-to-br from-green-500/10 to-transparent">
                    <div class="text-[10px] text-slate-500 font-orbitron uppercase mb-2">Faturamento Real (Mês)</div>
                    <div class="text-3xl font-bold font-orbitron text-green-400" id="fin-total-pago">R$ 0,00</div>
                    <div class="text-[9px] text-green-500/50 mt-2" id="fin-qtd-pago">0 itens liquidados</div>
                </div>
                <div class="nexus-card bg-gradient-to-br from-orange-500/10 to-transparent">
                    <div class="text-[10px] text-slate-500 font-orbitron uppercase mb-2">Pendente de Recebimento</div>
                    <div class="text-3xl font-bold font-orbitron text-orange-400" id="fin-total-pendente">R$ 0,00</div>
                    <div class="text-[9px] text-orange-500/50 mt-2" id="fin-qtd-pendente">0 itens em aberto</div>
                </div>
                <div class="nexus-card bg-gradient-to-br from-cyan-500/10 to-transparent">
                    <div class="text-[10px] text-slate-500 font-orbitron uppercase mb-2">Ticket Médio</div>
                    <div class="text-3xl font-bold font-orbitron text-cyan-400" id="fin-ticket-medio">R$ 0,00</div>
                    <div class="text-[9px] text-cyan-500/50 mt-2">Média por pagamento</div>
                </div>
            </div>

            <div class="nexus-card">
                <h6 class="font-orbitron text-cyan-400 mb-4 flex items-center gap-2">
                    <i class="fas fa-list-check"></i> Itens Financeiros Recentes
                </h6>
                <div class="table-responsive">
                    <table class="nexus-table">
                        <thead>
                            <tr>
                                <th>Placa</th>
                                <th>Descrição do Débito</th>
                                <th>Valor</th>
                                <th>Data</th>
                                <th class="text-end">Status</th>
                            </tr>
                        </thead>
                        <tbody id="finance-table-body">
                            <tr><td colspan="5" class="text-center py-10 opacity-50">Carregando dados financeiros...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        <!-- CONFIGURAÇÃO GLOBAL -->
        <div id="page-settings" class="page-content">
            <h2 class="font-orbitron text-cyan-400 text-lg mb-6 uppercase tracking-wider">Configuração Global do Sistema</h2>
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <!-- INFRAESTRUTURA -->
                <div class="nexus-card p-6">
                    <h6 class="font-orbitron text-xs text-cyan-400 mb-4 border-b border-white/5 pb-2">Infraestrutura e Automação</h6>
                    <div class="space-y-4">
                        <div>
                            <label class="block text-[10px] uppercase opacity-50 mb-1">CapSolver API Key (Captcha)</label>
                            <input type="text" id="global-capsolver-key" class="form-control bg-dark text-white border-white/10 font-mono text-xs">
                        </div>
                        <div>
                            <label class="block text-[10px] uppercase opacity-50 mb-1">API DH Key (Dados Técnicos)</label>
                            <input type="text" id="global-dh-key" class="form-control bg-dark text-white border-white/10 font-mono text-xs">
                        </div>
                    </div>
                </div>

                <!-- PROXY -->
                <div class="nexus-card p-6">
                    <h6 class="font-orbitron text-xs text-purple-400 mb-4 border-b border-white/5 pb-2">Configuração de Proxy (NodeMaven)</h6>
                    <div class="space-y-4">
                        <div>
                            <label class="block text-[10px] uppercase opacity-50 mb-1">Proxy Host</label>
                            <input type="text" id="global-proxy-host" class="form-control bg-dark text-white border-white/10 font-mono text-xs">
                        </div>
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label class="block text-[10px] uppercase opacity-50 mb-1">Proxy User</label>
                                <input type="text" id="global-proxy-user" class="form-control bg-dark text-white border-white/10 font-mono text-xs">
                            </div>
                            <div>
                                <label class="block text-[10px] uppercase opacity-50 mb-1">Proxy Pass</label>
                                <input type="password" id="global-proxy-pass" class="form-control bg-dark text-white border-white/10 font-mono text-xs">
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- API KEYS (LOCAL) -->
            <div class="nexus-card p-6 mt-6">
                <div class="flex justify-between items-center mb-4 border-b border-white/5 pb-2">
                    <h6 class="font-orbitron text-xs text-green-400 m-0"><i class="fas fa-key me-2"></i>API Keys - Placa API (Local)</h6>
                    <button class="btn btn-xs btn-outline-green font-orbitron" onclick="Settings.createKey()"><i class="fas fa-plus me-1"></i> NOVA KEY</button>
                </div>
                <div class="table-responsive">
                    <table class="table table-sm table-dark text-[10px]">
                        <thead>
                            <tr class="opacity-50">
                                <th>KEY VALUE</th>
                                <th>IDENTIFICAÇÃO</th>
                                <th>USO</th>
                                <th>EXPIRAÇÃO</th>
                                <th class="text-end">STATUS</th>
                            </tr>
                        </thead>
                        <tbody id="settings-keys-body">
                            <tr><td colspan="5" class="text-center py-4 opacity-50">Carregando chaves...</td></tr>
                        </tbody>
                    </table>
                </div>
            </div>

            <div class="mt-6 flex justify-end">
                <button class="btn btn-primary px-10 py-3 font-orbitron text-xs uppercase" onclick="Settings.save()">
                    <i class="fas fa-save me-2"></i>Salvar Tudo e Aplicar
                </button>
            </div>
        </div>
    </main>

    <!-- Modal Dossiê Detalhado -->
    <div class="modal fade" id="modal-details" tabindex="-1" aria-hidden="true" data-bs-backdrop="static">
        <div class="modal-dialog modal-fullscreen-sm-down modal-lg modal-dialog-centered">
            <div class="modal-content nexus-card" id="modal-details-body" style="padding:0; border: 1px solid rgba(0, 229, 255, 0.3)"></div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/api.js?v=<?php echo time(); ?>"></script>
    <script src="assets/js/auth.js?v=<?php echo time(); ?>"></script>
    <script src="assets/js/sites.js?v=<?php echo time(); ?>"></script>
    <script src="assets/js/users.js?v=<?php echo time(); ?>"></script>
    <script src="assets/js/pix.js?v=<?php echo time(); ?>"></script>
    
    <script>
        const App = {
            currentData: [],
            toggleSidebar() { $('#sidebar').toggleClass('open'); },
            showPage(pageId) {
                $('.page-content').removeClass('active'); $('#page-' + pageId).addClass('active');
                $('.menu-link').removeClass('active'); $(`.menu-link[data-page="${pageId}"]`).addClass('active');
                $('#sidebar').removeClass('open');
                if(pageId === 'dashboard') TabManager.load();
                if(pageId === 'analytics') AnalyticsManager.load();
                if(pageId === 'finance') FinanceManager.load();
                if(pageId === 'sites') Sites.load(true);
                if(pageId === 'users') Users.load(true);
                if(pageId === 'pix') { PIX.updateSiteSelector(); PIX.loadAllConfigs(); }
                if(pageId === 'automation') Automation.loadLogs();
            }
        };

        const Automation = {
            currentPage: 1,
            async loadLogs(page = 1) {
                this.currentPage = page;
                const tbody = $('#automation-logs-body');
                const dateFilter = $('#security-calendar').val() || ''; // Pega a data do calendário de segurança
                
                tbody.html('<tr><td colspan="5" class="text-center py-4 opacity-30">SINCRONIZANDO...</td></tr>');
                
                try {
                    const res = await Api.get(`list-data&type=blocked&page=${page}&date_start=${dateFilter}&date_end=${dateFilter}`);
                    if (!res.data || res.data.length === 0) {
                        tbody.html('<tr><td colspan="5" class="text-center py-6 opacity-30">Fim dos registros para este período</td></tr>');
                        return;
                    }

                    tbody.html(res.data.map(i => {
                        const date = new Date(i.updated_at).toLocaleString('pt-BR');
                        let typeColor = 'text-red-400';
                        let typeLabel = i.type;
                        
                        if (i.type === 'devtools_access' || i.type === 'devtools') { typeColor = 'text-yellow-400'; typeLabel = 'TESTE DEV'; }
                        if (i.type === 'bot') { typeColor = 'text-red-400'; typeLabel = 'BOT / SPIDER'; }
                        if (i.type === 'proxy_usage') { typeColor = 'text-blue-400'; typeLabel = 'USO PROXY'; }
                        if (i.type === 'captcha_resolved') { typeColor = 'text-purple-400'; typeLabel = 'CAPTCHA OK'; }

                        return `<tr class="border-b border-white/5 hover:bg-white/5 transition-all">
                            <td class="py-2"><span class="opacity-50 text-[9px] uppercase">${i.site_name}</span></td>
                            <td class="font-mono text-[9px]">${i.ip_address}</td>
                            <td class="font-bold ${typeColor}">${typeLabel}</td>
                            <td class="text-slate-400">${i.geo_city || '-'}/${i.geo_state || '-'}</td>
                            <td class="opacity-30 text-[9px] font-mono">${date}</td>
                        </tr>`;
                    }).join(''));

                    // Adicionar controles de paginação se houver dados
                    if (res.data.length === 10 || page > 1) {
                        tbody.append(`
                            <tr>
                                <td colspan="5" class="text-center py-3">
                                    <div class="flex justify-center gap-4">
                                        <button class="btn btn-xs btn-outline-secondary" ${page <= 1 ? 'disabled' : ''} onclick="Automation.loadLogs(${page - 1})">ANTERIOR</button>
                                        <span class="text-[10px] opacity-50 pt-1">PÁGINA ${page}</span>
                                        <button class="btn btn-xs btn-outline-secondary" ${res.data.length < 10 ? 'disabled' : ''} onclick="Automation.loadLogs(${page + 1})">PRÓXIMA</button>
                                    </div>
                                </td>
                            </tr>
                        `);
                    }
                } catch (e) {
                    tbody.html('<tr><td colspan="5" class="text-center py-4 text-red-500">Erro ao carregar logs</td></tr>');
                }
            }
        };

        const AnalyticsManager = {
            charts: {},
            async load() {
                const res = await Api.get('traffic-intelligence');
                if (res.status === 'success') {
                    const d = res.data;
                    this.renderInsights(d.insights);
                    this.renderRanking(d.sites);
                    this.renderTrends(d.history);
                    this.renderFunnel(d.stats);
                    $('#ai-forecast').text(Format.brl(d.forecast));
                }
            },
            renderInsights(insights) {
                const container = $('#ai-insights-container');
                if (!insights.length) {
                    container.html('<div class="text-[10px] opacity-30 italic">Analisando dados em tempo real...</div>');
                    return;
                }
                container.html(insights.map(i => `
                    <div class="p-3 rounded-xl border border-${i.type === 'danger' ? 'red' : (i.type === 'warning' ? 'orange' : 'green')}-500/20 bg-${i.type === 'danger' ? 'red' : (i.type === 'warning' ? 'orange' : 'green')}-500/5 text-[10px] flex gap-3">
                        <i class="fas fa-${i.type === 'danger' ? 'exclamation-circle' : 'lightbulb'} mt-1 text-${i.type === 'danger' ? 'red' : (i.type === 'warning' ? 'orange' : 'green')}-400"></i>
                        <span class="text-slate-300">${i.msg}</span>
                    </div>
                `).join(''));
            },
            renderRanking(sites) {
                const container = $('#site-ranking-body');
                container.html(sites.map((s, idx) => {
                    const convRate = s.visits > 0 ? ((s.conversions / s.visits) * 100).toFixed(1) : 0;
                    return `
                        <div class="flex items-center gap-4">
                            <div class="font-orbitron text-xs text-slate-600">#${idx+1}</div>
                            <div class="flex-1">
                                <div class="flex justify-between mb-1">
                                    <span class="text-[10px] font-bold text-white uppercase">${s.name}</span>
                                    <span class="text-[9px] text-cyan-400 font-mono">${Format.brl(s.revenue)}</span>
                                </div>
                                <div class="w-full h-1 bg-white/5 rounded-full overflow-hidden">
                                    <div class="h-full bg-cyan-500 shadow-[0_0_5px_#00e5ff]" style="width: ${convRate > 100 ? 100 : convRate}%"></div>
                                </div>
                                <div class="flex justify-between mt-1 text-[8px] opacity-50 uppercase">
                                    <span>${s.visits} Visitas</span>
                                    <span>${convRate}% Conv.</span>
                                </div>
                            </div>
                        </div>
                    `;
                }).join(''));
            },
            renderTrends(history) {
                if (this.charts.trends) this.charts.trends.destroy();
                const ctx = document.getElementById('conversionChart').getContext('2d');
                this.charts.trends = new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: history.map(h => h.data.split('-').reverse().join('/')),
                        datasets: [{
                            label: 'Faturamento (R$)',
                            data: history.map(h => h.revenue),
                            borderColor: '#00e5ff',
                            backgroundColor: 'rgba(0, 229, 255, 0.1)',
                            fill: true,
                            tension: 0.4
                        }]
                    },
                    options: { 
                        responsive: true, 
                        plugins: { legend: { display: false } },
                        scales: { y: { beginAtZero: true, grid: { color: 'rgba(255,255,255,0.05)' } }, x: { grid: { display: false } } }
                    }
                });
            },
            renderFunnel(stats) {
                if (this.charts.funnel) this.charts.funnel.destroy();
                const ctx = document.getElementById('funnelChart').getContext('2d');
                this.charts.funnel = new Chart(ctx, {
                    type: 'doughnut',
                    data: {
                        labels: ['Visitas', 'Consultas', 'PIX Gerados', 'Pagos'],
                        datasets: [{
                            data: [stats.visits, stats.consultas, stats.gerados, stats.pagos],
                            backgroundColor: ['rgba(255,255,255,0.1)', '#3b82f6', '#a855f7', '#22c55e'],
                            borderWidth: 0,
                            cutout: '80%'
                        }]
                    },
                    options: { 
                        responsive: true, 
                        plugins: { 
                            legend: { position: 'bottom', labels: { color: '#64748b', boxWidth: 8, font: { size: 9, family: 'Orbitron' } } } 
                        } 
                    }
                });
            }
        };

        const FinanceManager = {
            async load() {
                const res = await Api.get('financial-report');
                if (res.status === 'success') {
                    const s = res.data.summary;
                    $('#fin-total-pago').text(Format.brl(s.total_pago));
                    $('#fin-total-pendente').text(Format.brl(s.total_pendente));
                    $('#fin-qtd-pago').text(`${s.qtd_pagos} itens liquidados`);
                    $('#fin-qtd-pendente').text(`${s.qtd_pendentes} itens em aberto`);
                    
                    const ticket = s.qtd_pagos > 0 ? (s.total_pago / s.qtd_pagos) : 0;
                    $('#fin-ticket-medio').text(Format.brl(ticket));

                    this.loadRecent();
                }
            },
            async loadRecent() {
                // Reaproveitamos a busca de todos os débitos
                const res = await Api.get('list-data&type=pix_confirm');
                const tbody = $('#finance-table-body');
                
                if (!res.data || res.data.length === 0) {
                    tbody.html('<tr><td colspan="5" class="text-center py-10 opacity-50">Nenhuma transação recente</td></tr>');
                    return;
                }

                tbody.html(res.data.map(i => `
                    <tr class="hover:bg-white/5 border-b border-white/5">
                        <td class="py-3 font-bold text-cyan-400 font-mono">${i.placa || '-'}</td>
                        <td><span class="text-[10px] text-slate-400">${i.site_name}</span></td>
                        <td class="font-mono text-white">${Format.brl(i.valor)}</td>
                        <td class="text-[10px] opacity-50">${new Date(i.updated_at).toLocaleString('pt-BR')}</td>
                        <td class="text-end">
                            <span class="badge ${i.status_pagamento === 'pix_confirmado' ? 'bg-green-500/20 text-green-400' : 'bg-blue-500/20 text-blue-400'} border border-current text-[8px] uppercase">
                                ${i.status_pagamento}
                            </span>
                        </td>
                    </tr>
                `).join(''));
            }
        };

        const TabManager = {
            current: 'visits',
            currentPage: 1,
            switch(type, el) {
                if(el) { $('.nav-link').removeClass('active'); $(el).addClass('active'); }
                this.current = type; this.currentPage = 1; this.load();
            },
            async load(page = 1) {
                this.currentPage = page;
                $('#table-body').html('<tr><td colspan="10" class="text-center py-20 opacity-30 font-orbitron text-[10px] tracking-widest"><div class="spinner-border text-cyan-400 mb-3 block mx-auto w-6 h-6"></div>SINCRONIZANDO...</td></tr>');
                
                const range = $('#range-selector').val() || 'today';
                const search = $('#search-input').val() || '';
                const dStart = $('#date-start').val() || '';
                const dEnd = $('#date-end').val() || '';
                
                // Sincroniza estatísticas junto com a lista
                this.refreshStats(range);

                const res = await Api.get(`list-data&type=${this.current}&range=${range}&search=${search}&date_start=${dStart}&date_end=${dEnd}&page=${page}`);
                App.currentData = res.data || [];
                this.render(App.currentData);
            },
            render(data) {
                const head = $('#table-head'); const body = $('#table-body');
                const renderPagination = (listData) => {
                    if (listData.length === 10 || this.currentPage > 1) {
                        return `
                            <tr>
                                <td colspan="10" class="text-center py-3">
                                    <div class="flex justify-center gap-4">
                                        <button class="btn btn-xs btn-outline-info" ${this.currentPage <= 1 ? 'disabled' : ''} onclick="TabManager.load(${this.currentPage - 1})">ANTERIOR</button>
                                        <span class="text-[10px] opacity-50 pt-1">PÁGINA ${this.currentPage}</span>
                                        <button class="btn btn-xs btn-outline-info" ${listData.length < 10 ? 'disabled' : ''} onclick="TabManager.load(${this.currentPage + 1})">PRÓXIMA</button>
                                    </div>
                                </td>
                            </tr>
                        `;
                    }
                    return '';
                };

                if (this.current === 'visits') {
                    head.html('<tr><th>Site</th><th>IP</th><th>Device</th><th>Local</th><th>Data/Hora</th><th>Ações</th></tr>');
                    body.html(data.map(i => {
                        const countBadge = i.ip_visit_count > 1 ? `<span class="badge bg-danger ms-1" title="${i.ip_visit_count} visitas deste IP">${i.ip_visit_count}</span>` : '';
                        const date = new Date(i.visit_at || i.created_at).toLocaleString('pt-BR');
                        
                        let deviceIcon = 'desktop';
                        if (i.device === 'Mobile') deviceIcon = 'mobile-screen';
                        else if (i.device === 'devtools') deviceIcon = 'terminal text-yellow-400';

                        return `<tr>
                            <td><span class="text-cyan-400 font-bold">${i.site_name}</span>${countBadge}</td>
                            <td class="font-mono text-[10px]">${i.ip_address}</td>
                            <td><i class="fas fa-${deviceIcon}"></i></td>
                            <td class="text-[10px]">${i.geo_city || '-'}/${i.geo_state || '-'}</td>
                            <td class="text-[10px]">${date}</td>
                            <td><button onclick="View.details(${i.id})" class="text-info"><i class="fas fa-search-plus"></i></button></td>
                        </tr>`;
                    }).join('') + renderPagination(data));
                } else if (this.current === 'blocked') {
                    head.html('<tr><th>Site</th><th>IP</th><th>Tipo / Categoria</th><th>Local</th><th>Data/Hora</th><th>Ações</th></tr>');
                    body.html(data.map(i => {
                        const date = new Date(i.updated_at).toLocaleString('pt-BR');
                        let typeColor = 'text-red-400';
                        let typeLabel = i.type;
                        if (i.type === 'devtools_access' || i.type === 'devtools') { typeColor = 'text-yellow-400'; typeLabel = 'TESTE DEV'; }
                        if (i.type === 'bot') { typeColor = 'text-red-400'; typeLabel = 'BOT / SPIDER'; }
                        if (i.type === 'proxy_usage') { typeColor = 'text-blue-400'; typeLabel = 'USO PROXY'; }

                        return `<tr>
                            <td><span class="text-xs opacity-50">${i.site_name}</span></td>
                            <td class="font-mono text-[10px]">${i.ip_address}</td>
                            <td class="text-[10px]">
                                <span class="${typeColor} font-bold">${typeLabel}</span><br>
                                <span class="opacity-50 text-[9px] uppercase">${i.category}</span>
                            </td>
                            <td class="text-[10px]">${i.geo_city || '-'}/${i.geo_state || '-'}</td>
                            <td class="text-[10px]">${date}</td>
                            <td><button onclick="View.details(${i.id}, 'event')" class="text-info"><i class="fas fa-shield-alt"></i></button></td>
                        </tr>`;
                    }).join('') + renderPagination(data));
                } else {
                    head.html('<tr><th>Site</th><th>Placa</th><th>Total Déb</th><th>PIX</th><th>Data/Hora</th><th>Ações</th></tr>');
                    body.html(data.map(i => {
                        const date = new Date(i.query_at || i.updated_at).toLocaleString('pt-BR');
                        // Usar valores consolidados se disponíveis, caso contrário fallback para o campo da linha
                        const displayTotal = i.real_total_debt > 0 ? i.real_total_debt : i.total_debitos;
                        const displayPaid = i.accumulated_paid > 0 ? i.accumulated_paid : i.valor;
                        
                        return `<tr>
                            <td><span class="text-xs opacity-50">${i.site_name}</span></td>
                            <td><b>${i.placa||'-'}</b></td>
                            <td class="text-[11px]">${Format.brl(displayTotal)}</td>
                            <td class="text-warning font-bold text-[11px]">${Format.brl(displayPaid)}</td>
                            <td class="text-[10px]">${date}</td>
                            <td><button onclick="View.details(${i.id})" class="text-info"><i class="fas fa-eye" title="Ver dados completos"></i></button></td>
                        </tr>`;
                    }).join('') + renderPagination(data));
                }
            },
            async clearAll() {
                const { isConfirmed } = await Swal.fire({ title: 'Limpar Tudo?', text: 'Isso apagará permanentemente todos os registros.', icon: 'warning', showCancelButton: true, confirmButtonText: 'LIMPAR', confirmButtonColor: '#ef4444', background: '#0a101a', color: '#fff' });
                if (isConfirmed) { await Api.post('clear-logs'); this.load(); this.refreshStats(); Swal.fire('Limpo!', '', 'success'); }
            },
            async refreshStats(range = 'today') {
                const res = await Api.get(`dashboard&range=${range}`); 
                const s = res.data.summary;
                
                // Preencher valores dos cards (Período vs Total)
                $('#stat-visits-today').text(s.visits_period || 0);
                $('#stat-visits-total').text(s.visits_total || 0);
                
                $('#stat-consultas-today').text(s.consultas_period || 0);
                $('#stat-consultas-total').text(s.consultas_total || 0);
                
                $('#stat-pix-today').text(s.pix_period || 0);
                $('#stat-pix-total').text(s.pix_total || 0);
                
                $('#stat-pix-value-today').text(Format.brl(s.pix_value_period));
                $('#stat-pix-value-total').text(Format.brl(s.pix_value_total));
                
                // Novos contadores de segurança
                $('#stat-bots').text(s.bots_period || 0);
                $('#stat-redirects').text(s.redirect_period || 0);
                $('#stat-ddos').text(s.ddos_period || 0);

                // Saúde do sistema e APIs
                const apis = ['zapay', 'ms', 'mg', 'rj'];
                apis.forEach(api => {
                    const el = $('#status-api-' + api);
                    if (s['api_' + api]) {
                        el.text('ON').addClass('text-green-400').removeClass('text-red-500');
                    } else {
                        el.text('OFF').addClass('text-red-500').removeClass('text-green-400');
                    }
                });

                $('#stat-solved-captcha').text(s.solved_captcha || 0);
                $('#stat-proxy-requests').text(s.proxy_requests || 0);
                $('#stat-devtools').text(s.devtools_period || 0);
                
                $('#stat-proxy-period').text(s.proxy_period || 0);
                $('#stat-proxy-total').text(s.proxy_total || 0);

                // Saúde Captcha e Proxy
                $('#stat-solved-captcha').text(s.solved_captcha || 0);
                $('#stat-total-solved-captcha').text(s.total_solved_captcha || 0);
                $('#stat-proxy-requests').text(s.proxy_requests || 0);
                $('#stat-total-proxy-requests').text(s.total_proxy_requests || 0);
                
                $('#status-captcha').text(`SALDO: $ ${parseFloat(s.captcha_balance || 0).toFixed(4)}`).addClass('text-purple-400');
                
                // Segurança e Bloqueios
                $('#stat-bots').text(s.bots_period || 0);
                $('#stat-bots-total').text(`TOTAL: ${s.bots_total || 0}`);
                $('#stat-redirects').text(s.redirect_period || 0);
                $('#stat-redirects-total').text(`TOTAL: ${s.redirect_total || 0}`);
                $('#stat-ddos').text(s.ddos_period || 0);
                $('#stat-ddos-total').text(`TOTAL: ${s.ddos_total || 0}`);

                // Sincronizar Calendário
                if (s.active_security_dates) {
                    SecurityCalendar.highlightDates(s.active_security_dates);
                }
                $('#status-proxy').text(s.proxy_requests > 0 ? 'HEALTH: ACTIVE' : 'HEALTH: STANDBY').addClass('text-cyan-400');
                
                $('#server-time').text(new Date().toLocaleTimeString('pt-BR', {hour:'2-digit', minute:'2-digit'}));
            },
            export() {
                if (!App.currentData.length) return Swal.fire('Sem dados', 'Não há nada para exportar nesta aba.', 'info');
                let csv = 'Site,Data,IP,Placa,ValorTotal,ValorPIX,Local\n';
                App.currentData.forEach(i => {
                    csv += `"${i.site_name}","${i.visit_at || i.created_at}","${i.ip_address}","${i.placa || '-'}","${i.total_debitos || 0}","${i.valor || 0}","${i.geo_city}"\n`;
                });
                const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
                const link = document.createElement("a");
                link.href = URL.createObjectURL(blob); link.setAttribute("download", `nexus_export_${this.current}.csv`);
                document.body.appendChild(link); link.click(); document.body.removeChild(link);
            }
        };

        const View = {
            async details(id) {
                const res = await Api.get(`get-consulta&id=${id}`);
                const c = res.data; 
                const meta = typeof c.metadata === 'string' ? JSON.parse(c.metadata) : (c.metadata || {});
                
                // Lógica de Inferência do Funil
                const pixGenAt = c.pix_gen_at || c.pix_copy_at || c.confirm_at;
                
                const history = [
                    { l: 'Visita', t: c.visit_at },
                    { l: 'Consulta', t: c.query_at },
                    { l: 'Gerou PIX', t: pixGenAt },
                    { l: 'Copiou', t: c.pix_copy_at },
                    { l: 'Confirmou', t: c.confirm_at }
                ];

                const v = meta.veiculo || meta.veiculo_dados || meta.vehicle || {};
                
                // Capturar itens selecionados no último PIX (se disponível no metadata)
                const selectedItems = meta.extrato || [];
                const hasPixInteraction = pixGenAt || c.pix_copy_at || c.confirm_at;

                let accessCount = 0;
                let realTotalDebt = 0;
                try {
                    // Buscar contagem real de acessos para este IP/Placa (Evita erro de contagem geral se placa vazia)
                    const searchParam = c.placa ? c.placa : c.ip_address;
                    const statsRes = await Api.get(`list-data&type=consultas&search=${searchParam}`);
                    accessCount = statsRes.data.filter(i => (c.placa && i.placa === c.placa) || (i.ip_address === c.ip_address)).length;
                    
                    // Buscar débitos granulares para somar a Dívida Total Real (Pendentes + Pagos)
                    if (c.placa) {
                        const finRes = await Api.get(`get-finance&placa=${c.placa}`);
                        if (finRes.data && finRes.data.length > 0) {
                            realTotalDebt = finRes.data.reduce((acc, curr) => acc + parseFloat(curr.valor), 0);
                        } else {
                            realTotalDebt = parseFloat(c.total_debitos || 0);
                        }
                    } else {
                        realTotalDebt = 0;
                    }
                } catch(e) {
                    realTotalDebt = parseFloat(c.total_debitos || 0);
                }

                let html = `
                    <div class="modal-header border-b border-white/5 p-4 bg-gradient-to-r from-cyan-500/10 to-transparent">
                        <h5 class="modal-title font-orbitron text-cyan-400 text-sm flex items-center gap-2">
                            <i class="fas fa-file-invoice"></i> DOSSIÊ: ${c.placa || 'Visitante'} [${c.site_name}]
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body p-4">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <h6 class="text-cyan-400 font-orbitron text-[10px] uppercase mb-3 flex items-center gap-2">
                                    <i class="fas fa-network-wired"></i> Informações Técnicas
                                </h6>
                                <div class="bg-black/40 p-4 rounded-2xl border border-white/5 mb-4 shadow-inner">
                                    <p class="text-[11px] mb-2 flex justify-between"><b>IP:</b> <span class="text-cyan-400">${c.ip_address}</span></p>
                                    <p class="text-[11px] mb-2 flex justify-between"><b>Local:</b> <span>${c.geo_city || '-'} / ${meta.regionName || meta.region || meta.geo_state || '-'}</span></p>
                                    <p class="text-[11px] mb-2 flex justify-between"><b>Provedor:</b> <span>${meta.isp || meta.ip_details?.isp || '-'}</span></p>
                                    <div class="text-[9px] text-slate-500 mt-3 p-2 bg-white/5 rounded border border-white/5 break-all font-mono">${c.user_agent}</div>
                                </div>

                                <h6 class="text-cyan-400 font-orbitron text-[10px] uppercase mb-3 flex items-center gap-2">
                                    <i class="fas fa-car"></i> Dados do Veículo
                                </h6>
                                <div class="bg-blue-500/5 p-4 rounded-2xl border border-blue-500/20">
                                    <p class="text-[11px] mb-2 flex justify-between border-b border-white/5 pb-1"><b>Proprietário:</b> <span class="text-white font-semibold">${c.proprietario || v.proprietario || v.nome || meta.nome || '-'}</span></p>
                                    <p class="text-[11px] mb-2 flex justify-between border-b border-white/5 pb-1"><b>Modelo:</b> <span class="text-white font-semibold">${c.modelo || v.model || v.marcaModelo || v.veiculo || '-'}</span></p>
                                    <p class="text-[11px] mb-2 flex justify-between border-b border-white/5 pb-1"><b>Renavam:</b> <span class="text-white font-semibold font-mono">${c.renavam || v.renavam || '-'}</span></p>
                                    <p class="text-[11px] mb-2 flex justify-between"><b>Cor/Ano:</b> <span>${v.color || v.cor || '-'} / ${v.year || v.anoFabricacao || '-'}</span></p>
                                </div>
                            </div>

                            <div class="col-md-6 border-l border-white/5">
                                <ul class="nav nav-pills mb-4 bg-black/20 p-1 rounded-xl border border-white/5 flex" id="details-tabs">
                                    <li class="flex-1"><button class="w-full text-[9px] font-orbitron py-2 active" data-bs-toggle="pill" data-bs-target="#tab-funnel">FUNIL</button></li>
                                    <li class="flex-1"><button class="w-full text-[9px] font-orbitron py-2" data-bs-toggle="pill" data-bs-target="#tab-finance" onclick="${c.placa ? `View.loadFinance('${c.placa}')` : ''}" ${!c.placa ? 'disabled opacity-50' : ''}>PENDENTES</button></li>
                                </ul>

                                <div class="tab-content">
                                    <div class="tab-pane fade show active" id="tab-funnel">
                                        <h6 class="text-cyan-400 font-orbitron text-[10px] uppercase mb-3 flex items-center gap-2">
                                            <i class="fas fa-stream"></i> Funil de Conversão
                                        </h6>
                                        <div class="ps-4 border-l-2 border-white/5 ms-2 mb-4 space-y-4">
                                            ${history.map(h => `
                                                <div class="position-relative">
                                                    <span class="position-absolute start-0 translate-middle p-1 rounded-circle ${h.t ? 'bg-cyan-400 shadow-[0_0_8px_#00e5ff] scale-110' : 'bg-slate-800'}" style="left: -17px !important; margin-top: 5px; transition: 0.3s"></span>
                                                    <div class="text-[10px] font-bold ${h.t ? 'text-cyan-300' : 'text-slate-600'} uppercase tracking-wider">${h.l}</div>
                                                    <div class="text-[9px] font-mono ${h.t ? 'text-slate-300' : 'text-slate-500'}">${h.t ? new Date(h.t).toLocaleString('pt-BR') : 'Pendente...'}</div>
                                                </div>
                                            `).join('')}
                                        </div>

                                        ${hasPixInteraction && selectedItems.length > 0 ? `
                                            <div class="mt-4 p-3 bg-cyan-500/5 border border-cyan-500/10 rounded-xl">
                                                <div class="text-[9px] text-cyan-400 font-orbitron uppercase mb-2">Itens no último PIX:</div>
                                                <div class="space-y-1 max-h-[120px] overflow-y-auto pr-2 custom-scrollbar">
                                                    ${selectedItems.map(item => `
                                                        <div class="flex justify-between text-[9px] border-b border-white/5 pb-1">
                                                            <span class="text-slate-400 truncate mr-2">${item.descricao || item.title || 'Débito'}</span>
                                                            <span class="text-white font-mono">R$ ${parseFloat(item.valor || item.amount).toLocaleString('pt-BR', {minimumFractionDigits:2})}</span>
                                                        </div>
                                                    `).join('')}
                                                </div>
                                            </div>
                                        ` : ''}
                                    </div>

                                    <div class="tab-pane fade" id="tab-finance">
                                        <h6 class="text-cyan-400 font-orbitron text-[10px] uppercase mb-3 flex items-center gap-2">
                                            <i class="fas fa-clock"></i> Itens Pendentes de Pagamento
                                        </h6>
                                        <div class="bg-black/20 rounded-xl border border-white/5 overflow-hidden">
                                            <div class="table-responsive" style="max-height: 300px; overflow-y: auto;">
                                                <table class="table table-sm table-dark text-[10px] mb-0">
                                                    <thead class="bg-white/5 sticky top-0"><tr><th>Descrição</th><th class="text-end">Status</th></tr></thead>
                                                    <tbody id="finance-items-body">
                                                        <tr><td colspan="2" class="text-center py-4 opacity-50">Sincronizando...</td></tr>
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer border-t border-white/5 p-4 bg-black/40">
                        <div class="w-100 flex flex-col sm:flex-row justify-between items-center gap-4">
                            <div class="flex gap-6">
                                <div class="flex flex-col">
                                    <span class="text-[9px] text-slate-500 uppercase font-orbitron">Dívida Total Real</span>
                                    <span class="text-white text-lg font-bold font-orbitron">${Format.brl(realTotalDebt)}</span>
                                </div>
                                <div class="flex flex-col border-l border-white/10 pl-6">
                                    <span class="text-[9px] text-green-500 uppercase font-orbitron">Total Pago Acumulado</span>
                                    <span class="text-green-400 text-lg font-bold font-orbitron">${Format.brl(c.valor)}</span>
                                </div>
                            </div>
                            <div class="flex gap-2">
                                <button type="button" class="btn-nexus bg-cyan-500/10 text-cyan-400 border border-cyan-500/20 px-4 py-2" onclick="View.showMoreDetails('${c.ip_address}', '${c.placa||'N/A'}', ${accessCount})">
                                    <i class="fas fa-info-circle"></i> MAIS DETALHES
                                </button>
                                <button type="button" class="btn-nexus bg-white/5 text-slate-400 border border-white/10 px-4 py-2" onclick="$('#modal-details').modal('hide')">FECHAR</button>
                            </div>
                        </div>
                    </div>
                `;
                $('#modal-details-body').html(html); 
                new bootstrap.Modal('#modal-details').show();
            },

            showMoreDetails(ip, placa, count) {
                Swal.fire({
                    title: 'HISTÓRICO DE ACESSO',
                    html: `
                        <div class="text-left font-mono text-xs space-y-2">
                            <p><b>IP Identificado:</b> ${ip}</p>
                            <p><b>Placa Consultada:</b> ${placa}</p>
                            <p class="text-cyan-400"><b>Total de Interações:</b> ${count} vezes</p>
                            <hr class="border-white/10 my-3">
                            <p class="text-[10px] text-slate-500 italic">O sistema consolida os pagamentos acumulados e filtra itens já quitados automaticamente.</p>
                        </div>
                    `,
                    background: '#0a101a',
                    color: '#fff',
                    confirmButtonText: 'FECHAR',
                    confirmButtonColor: '#00e5ff'
                });
            },

            async loadFinance(placa) {
                if (!placa) return;
                const body = $('#finance-items-body');
                try {
                    const res = await Api.get(`get-finance&placa=${placa}`);
                    if (!res.data || res.data.length === 0) {
                        body.html('<tr><td colspan="2" class="text-center py-4 opacity-50">Nenhum débito pendente.</td></tr>');
                        return;
                    }
                    const pendentes = res.data.filter(d => d.status === 'pendente');
                    if (pendentes.length === 0) {
                        body.html('<tr><td colspan="2" class="text-center py-4 text-green-400 font-bold uppercase"><i class="fas fa-check-circle"></i> Tudo Pago!</td></tr>');
                        return;
                    }
                    body.html(pendentes.map(d => `
                        <tr class="border-b border-white/5">
                            <td class="ps-3 py-2">
                                <div class="text-[10px] font-bold text-slate-300">${d.descricao}</div>
                                <div class="text-[9px] text-slate-500 font-mono">R$ ${parseFloat(d.valor).toLocaleString('pt-BR', {minimumFractionDigits:2})}</div>
                            </td>
                            <td class="text-end pe-3 py-2">
                                <span class="badge bg-orange-500/20 text-orange-400 border border-current text-[8px] uppercase tracking-tighter">${d.status}</span>
                            </td>
                        </tr>
                    `).join(''));
                } catch (e) {
                    body.html('<tr><td colspan="2" class="text-center py-4 text-red-400">Erro de sincronização.</td></tr>');
                }
            },

            async showProxyList() {
                // ... (existing code)
            },

            async showCaptchaLogs() {
                Swal.fire({
                    title: 'HISTÓRICO DE SALDO (CAPTCHA)',
                    width: '600px',
                    html: `
                        <div class="bg-black/20 rounded-xl border border-white/5 overflow-hidden text-left">
                            <table class="table table-sm table-dark text-[10px] mb-0">
                                <thead class="bg-white/5"><tr><th>Horário</th><th>Saldo ($)</th><th>Status</th></tr></thead>
                                <tbody id="captcha-logs-body">
                                    <tr><td colspan="3" class="text-center py-4">Carregando logs...</td></tr>
                                </tbody>
                            </table>
                        </div>
                    `,
                    background: '#0a101a',
                    color: '#fff',
                    showConfirmButton: false,
                    didOpen: async () => {
                        try {
                            const res = await Api.get('list-captcha-logs');
                            const body = $('#captcha-logs-body');
                            if (!res.data || res.data.length === 0) {
                                body.html('<tr><td colspan="3" class="text-center py-4 opacity-50">Nenhum log de captcha encontrado.</td></tr>');
                                return;
                            }
                            body.html(res.data.map(i => {
                                return `
                                    <tr class="border-b border-white/5">
                                        <td class="p-2 opacity-50 font-mono">${new Date(i.created_at).toLocaleString('pt-BR')}</td>
                                        <td class="p-2 text-purple-400 font-bold font-mono">$ ${parseFloat(i.balance).toFixed(4)}</td>
                                        <td class="p-2">
                                            <span class="badge ${i.status === 'success' ? 'bg-green-500/20 text-green-400' : 'bg-red-500/20 text-red-400'} text-[8px]">
                                                ${i.status.toUpperCase()}
                                            </span>
                                        </td>
                                    </tr>
                                `;
                            }).join(''));
                        } catch(e) {
                            $('#captcha-logs-body').html('<tr><td colspan="3" class="text-center py-4 text-red-400">Erro ao carregar lista.</td></tr>');
                        }
                    }
                });
            }
        };

        const Settings = {
            async load() {
                try {
                    const res = await Api.get('get-settings');
                    if (res.status === 'success') {
                        const s = res.data;
                        document.getElementById('global-capsolver-key').value = s.capsolver_key || '';
                        document.getElementById('global-dh-key').value = s.api_dh_key || '';
                        document.getElementById('global-proxy-host').value = s.proxy_host || '';
                        document.getElementById('global-proxy-user').value = s.proxy_user || '';
                        document.getElementById('global-proxy-pass').value = s.proxy_pass || '';
                    }
                    this.loadKeys();
                } catch(e) {}
            },
            async loadKeys() {
                const body = $('#settings-keys-body');
                try {
                    const res = await Api.get('api-keys');
                    if (res.status === 'success' && res.data) {
                        if (res.data.length === 0) {
                            body.html('<tr><td colspan="5" class="text-center py-4 opacity-50">Nenhuma key cadastrada.</td></tr>');
                            return;
                        }
                        body.html(res.data.map(k => `
                            <tr class="border-b border-white/5">
                                <td class="font-mono text-cyan-400">${k.key_value}</td>
                                <td class="opacity-70">${k.owner || '-'}</td>
                                <td>${k.total_requests} reqs</td>
                                <td class="opacity-50">${new Date(k.expires_at).toLocaleDateString('pt-BR')}</td>
                                <td class="text-end">
                                    <button class="btn btn-xs ${k.active ? 'btn-outline-success' : 'btn-outline-danger'}" onclick="Settings.toggleKey(${k.id})">
                                        ${k.active ? 'ATIVA' : 'INATIVA'}
                                    </button>
                                </td>
                            </tr>
                        `).join(''));
                    }
                } catch(e) {
                    body.html('<tr><td colspan="5" class="text-center py-4 text-red-400">Falha ao carregar keys.</td></tr>');
                }
            },
            async toggleKey(id) {
                try {
                    const res = await Api.post('toggle-api-key', { id });
                    if (res.status === 'success') {
                        this.loadKeys();
                    }
                } catch(e) {}
            },
            async createKey() {
                const { value: formValues } = await Swal.fire({
                    title: 'NOVA API KEY',
                    html:
                        '<input id="swal-input1" class="swal2-input" placeholder="Identificador (ex: DF Site)">' +
                        '<input id="swal-input2" class="swal2-input" type="date" value="2027-12-31">',
                    focusConfirm: false,
                    background: '#0a101a', color: '#fff',
                    preConfirm: () => {
                        return [
                            document.getElementById('swal-input1').value,
                            document.getElementById('swal-input2').value
                        ]
                    }
                });

                if (formValues) {
                    UI.showLoading('Gerando key...');
                    try {
                        const res = await Api.post('create-api-key', { identifier: formValues[0], expires_at: formValues[1] });
                        if (res.status === 'success') {
                            Swal.fire('Gerada!', `Sua nova key: ${res.data.key_value}`, 'success');
                            this.loadKeys();
                        }
                    } catch(e) {
                        Swal.fire('Erro', 'Falha ao criar key.', 'error');
                    } finally { UI.hideLoading(); }
                }
            },
            async save() {
                const payload = {
                    capsolver_key: document.getElementById('global-capsolver-key').value,
                    api_dh_key: document.getElementById('global-dh-key').value,
                    proxy_host: document.getElementById('global-proxy-host').value,
                    proxy_user: document.getElementById('global-proxy-user').value,
                    proxy_pass: document.getElementById('global-proxy-pass').value
                };
                UI.showLoading('Salvando configurações...');
                try {
                    const res = await Api.post('save-settings', payload);
                    if (res.status === 'success') {
                        Swal.fire('Sucesso!', 'Configurações globais atualizadas e aplicadas.', 'success');
                    }
                } catch(e) {
                    Swal.fire('Erro', 'Falha ao salvar.', 'error');
                } finally {
                    UI.hideLoading();
                }
            },
            async testCaptcha() {
                UI.showLoading('Consultando saldo...');
                try {
                    const res = await Api.post('test-captcha');
                    if (res.status === 'success') {
                        Swal.fire({
                            title: 'SALDO CAPSOLVER',
                            text: `Seu saldo atual é: $ ${parseFloat(res.data.balance).toFixed(4)}`,
                            icon: 'success',
                            background: '#0a101a', color: '#fff', confirmButtonColor: '#00e5ff'
                        });
                    }
                } catch(e) {
                    Swal.fire('Erro', 'Falha ao consultar captcha. Verifique a chave.', 'error');
                } finally { UI.hideLoading(); }
            },
            async testProxy() {
                UI.showLoading('Testando rotação de IP...');
                try {
                    const res = await Api.post('test-proxy');
                    if (res.status === 'success') {
                        Swal.fire({
                            title: 'PROXY ATIVA',
                            html: `<div class='text-center'><p class='mb-2'>Proxy funcionando corretamente!</p><p class='text-cyan-400 font-mono font-bold text-lg'>IP: ${res.data.ip}</p></div>`,
                            icon: 'success',
                            background: '#0a101a', color: '#fff', confirmButtonColor: '#00e5ff'
                        });
                    }
                } catch(e) {
                    Swal.fire('Erro', 'Falha na conexão proxy. Verifique as credenciais.', 'error');
                } finally { UI.hideLoading(); }
            }
        };

        const SecurityCalendar = {
            instance: null,
            init() {
                this.instance = flatpickr("#security-calendar", {
                    theme: "dark",
                    dateFormat: "Y-m-d",
                    onChange: (selectedDates, dateStr) => {
                        this.filterByDate(dateStr);
                    },
                    onDayCreate: (dObj, dStr, fp, dayElem) => {
                        // Formatar data local para comparação
                        const year = dayElem.dateObj.getFullYear();
                        const month = String(dayElem.dateObj.getMonth() + 1).padStart(2, '0');
                        const day = String(dayElem.dateObj.getDate()).padStart(2, '0');
                        const date = `${year}-${month}-${day}`;
                        
                        if (this.activeDates && this.activeDates.includes(date)) {
                            dayElem.classList.add("has-security-event");
                            dayElem.title = "Atividade Detectada";
                        }
                    }
                });
            },
            activeDates: [],
            highlightDates(dates) {
                this.activeDates = dates;
                if (this.instance) this.instance.redraw();
            },
            async filterByDate(date) {
                try {
                    const res = await Api.get(`dashboard&range=all&date_start=${date}&date_end=${date}`);
                    const s = res.data.summary;
                    $('#stat-bots').text(s.bots_period || 0);
                    $('#stat-redirects').text(s.redirect_period || 0);
                    $('#stat-ddos').text(s.ddos_period || 0);
                } catch(e) {}
            }
        };

        $(document).ready(() => { 
            TabManager.refreshStats(); TabManager.load(); Settings.load(); SecurityCalendar.init(); setInterval(() => TabManager.refreshStats(), 15000); 
            const space = document.getElementById("space-bg");
            for (let i = 0; i < 60; i++) {
                const star = document.createElement("div"); star.className = "star";
                star.style.left = Math.random() * 100 + "%"; star.style.top = Math.random() * 100 + "%";
                star.style.width = (Math.random() * 2 + 1) + "px"; star.style.height = star.style.width;
                space.appendChild(star);
            }
        });
    </script>
</body>
</html>