# Admin Server API - Estrutura Modular

## 📁 Estrutura de Arquivos

```
api/
├── admin.php              # Entry point principal
├── admin.php.backup       # Backup do arquivo original (1375 linhas)
│
├── core/                  # Classes utilitárias
│   ├── Request.php        # Helpers para requisições HTTP
│   ├── Response.php       # Helpers para respostas JSON
│   └── Router.php         # Roteador centralizado
│
├── middleware/            # Middleware de autenticação
│   └── Auth.php           # Autenticação e autorização
│
├── controllers/           # Lógica de negócio
│   ├── AuthController.php       # Login, logout, me
│   ├── SitesController.php      # CRUD de sites
│   ├── UsersController.php      # CRUD de usuários
│   ├── PixController.php        # Config e stats PIX
│   ├── DashboardController.php  # Dashboard e eventos
│   └── SettingsController.php   # API token, ping
│
└── v1/
    └── index.php          # API v1 (tracking)
```

## 🛣️ Rotas Disponíveis

### Autenticação
| Método | Action | Descrição |
|--------|--------|-----------|
| POST | `login` | Autenticar usuário |
| POST | `logout` | Encerrar sessão |
| GET | `me` | Info do usuário logado |
| GET | `permissions` | Listar permissões disponíveis |

### Sites
| Método | Action | Descrição |
|--------|--------|-----------|
| GET | `sites` | Listar todos os sites |
| POST | `sites` | Criar novo site (super admin) |
| GET | `site?id=X` | Detalhes de um site |
| PUT | `site?id=X` | Atualizar site |
| DELETE | `site?id=X` | Excluir site (super admin) |
| POST | `clear-site-events?site_id=X` | Limpar eventos do site |

### Usuários
| Método | Action | Descrição |
|--------|--------|-----------|
| GET | `users` | Listar todos os usuários |
| POST | `users` | Criar novo usuário |
| GET | `user?id=X` | Detalhes de um usuário |
| PUT | `user?id=X` | Atualizar usuário |
| DELETE | `user?id=X` | Excluir usuário |
| GET | `site-users?site_id=X` | Usuários de um site |

### PIX
| Método | Action | Descrição |
|--------|--------|-----------|
| GET | `pix-config?site_id=X` | Obter config PIX |
| POST | `pix-config?site_id=X` | Salvar config PIX |
| GET | `pix-history?site_id=X` | Histórico de alterações |
| GET | `pix-stats?site_id=X` | Estatísticas PIX |

### Dashboard
| Método | Action | Descrição |
|--------|--------|-----------|
| GET | `dashboard` | Estatísticas gerais |
| GET | `events` | Listar eventos (paginado) |
| GET | `event-types` | Tipos de eventos |

### Configurações
| Método | Action | Descrição |
|--------|--------|-----------|
| GET/POST | `api-token` | Token da API DETRAN |
| GET | `ping?url=X` | Testar conexão |

## 🔧 Uso das Classes

### Request (requisições)
```php
use Core\Request;

// Métodos HTTP
Request::method();        // GET, POST, PUT, DELETE
Request::isGet();         // boolean
Request::isPost();        // boolean

// Dados da requisição
Request::body();          // Array do JSON body
Request::input('key');    // Valor específico do body
Request::get('param');    // Query string (?param=value)

// Headers
Request::header('X-API-Key');
Request::bearerToken();   // Token do Authorization header
Request::apiKey();        // X-API-Key header

// Utilitários
Request::ip();            // IP do cliente
Request::userAgent();     // User Agent
Request::pagination();    // [page, limit, offset]
Request::validate(['field1', 'field2']);  // Validação
```

### Response (respostas)
```php
use Core\Response;

// Sucesso
Response::success($data);
Response::success($data, 'Mensagem');
Response::success(null, 'Operação concluída');

// Erro
Response::error('Mensagem de erro', 400);
Response::unauthorized();  // 401
Response::forbidden();     // 403
Response::notFound();      // 404
Response::serverError();   // 500

// Paginação
Response::paginated($data, $total, $page, $limit);

// JSON genérico
Response::json(['status' => 'custom', 'data' => $data], 200);
```

### Auth (autenticação)
```php
use Middleware\Auth;

// Obter usuário
$user = Auth::getUser();          // null se não autenticado
$user = Auth::require();          // Erro 401 se não autenticado

// Verificações
Auth::isSuperAdmin();             // boolean
Auth::hasAccessToSite($siteId);   // boolean
Auth::hasPermission('edit_pix');  // boolean

// Requerimentos (erros automáticos)
Auth::requireSuperAdmin();        // Erro 403 se não for super admin
Auth::requireSiteAccess($siteId); // Erro 403 se não tiver acesso
Auth::requirePermission('edit_pix');

// Helpers
Auth::getUserSiteId();            // null para super admin, int para user de site
```

## 📝 Criando um Novo Controller

```php
<?php
namespace Controllers;

use Core\Request;
use Core\Response;
use Middleware\Auth;

class NovoController
{
    public static function index(): void
    {
        $user = Auth::require();
        $pdo = getDBConnection();
        
        // Lógica...
        
        Response::success($data);
    }
    
    public static function create(): void
    {
        Auth::requireSuperAdmin();
        
        $nome = Request::input('nome', '');
        
        if (!$nome) {
            Response::error('Nome obrigatório', 400);
        }
        
        // Criar...
        
        Response::success(['id' => $id], 'Criado com sucesso');
    }
}
```

Adicione a rota no `Router.php`:
```php
'nova-rota' => [NovoController::class, 'index', 'GET'],
'nova-rota:create' => [NovoController::class, 'create', 'POST'],
```

## ✅ Benefícios da Modularização

1. **Manutenibilidade**: Código organizado por responsabilidade
2. **Testabilidade**: Cada controller pode ser testado isoladamente
3. **Legibilidade**: Arquivos menores e focados
4. **Escalabilidade**: Fácil adicionar novos controllers
5. **Reutilização**: Classes utilitárias compartilhadas
6. **Debug**: Mais fácil identificar erros

## 🔄 Migração

O arquivo original foi mantido como `admin.php.backup` para referência.
Todas as funcionalidades foram preservadas e os endpoints continuam os mesmos.
