<?php
/**
 * Settings Controller
 * Gerencia configurações globais (API Token, Ping, etc.)
 */

namespace Controllers;

use Core\Request;
use Core\Response;
use Middleware\Auth;

class SettingsController
{
    /**
     * GET/POST /api-token - Gerenciar token da API do DETRAN
     */
    public static function apiToken(): void
    {
        // Apenas super admin pode acessar
        $currentUser = Auth::requireSuperAdmin();
        
        $pdo = getDBConnection();

        // Criar tabela se não existir
        $pdo->exec("
            CREATE TABLE IF NOT EXISTS global_settings (
                id INT AUTO_INCREMENT PRIMARY KEY,
                setting_key VARCHAR(100) NOT NULL UNIQUE,
                setting_value TEXT NOT NULL,
                description VARCHAR(255) NULL,
                is_encrypted TINYINT(1) DEFAULT 0,
                updated_by INT NULL,
                created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
                updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                INDEX idx_key (setting_key)
            ) ENGINE=InnoDB
        ");

        if (Request::isPost()) {
            // Salvar token
            $token = trim(Request::input('token', ''));

            if (empty($token)) {
                Response::error('Token não pode ser vazio', 400);
            }

            // Validar formato básico do token (deve ser um bearer token longo)
            if (strlen($token) < 50) {
                Response::error('Token parece inválido (muito curto)', 400);
            }

            $stmt = $pdo->prepare("
                INSERT INTO global_settings (setting_key, setting_value, description, updated_by)
                VALUES ('detran_api_token', ?, 'Token Bearer para API do DETRAN RS', ?)
                ON DUPLICATE KEY UPDATE 
                    setting_value = VALUES(setting_value),
                    updated_by = VALUES(updated_by),
                    updated_at = NOW()
            ");
            $stmt->execute([$token, $currentUser['id']]);

            Response::success(null, 'Token salvo com sucesso');
        } else {
            // Buscar token atual
            $stmt = $pdo->prepare("
                SELECT gs.setting_value as token, gs.updated_at, u.username as updated_by
                FROM global_settings gs
                LEFT JOIN admin_users u ON u.id = gs.updated_by
                WHERE gs.setting_key = 'detran_api_token'
            ");
            $stmt->execute();
            $result = $stmt->fetch();

            if ($result) {
                Response::success([
                    'token' => $result['token'],
                    'updated_at' => $result['updated_at'],
                    'updated_by' => $result['updated_by']
                ]);
            } else {
                Response::success(null, 'Token não configurado');
            }
        }
    }

    /**
     * GET /ping?url={url} - Testar conexão com uma URL
     */
    public static function ping(): void
    {
        Auth::require();

        $url = Request::get('url', '');

        if (!$url) {
            Response::error('URL required', 400);
        }

        // Validar URL
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            Response::error('Invalid URL format', 400);
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 15,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 5,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 Admin-Panel-Ping/1.0',
            CURLOPT_HTTPHEADER => [
                'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8',
                'Accept-Language: pt-BR,pt;q=0.9,en;q=0.8'
            ],
            CURLOPT_NOBODY => false,
            CURLOPT_HEADER => true
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $totalTime = round(curl_getinfo($ch, CURLINFO_TOTAL_TIME) * 1000);
        $error = curl_error($ch);
        $errno = curl_errno($ch);
        curl_close($ch);

        if ($errno) {
            // Erro de conexão
            $errorMessages = [
                CURLE_COULDNT_RESOLVE_HOST => 'Não foi possível resolver o domínio (DNS)',
                CURLE_COULDNT_CONNECT => 'Não foi possível conectar ao servidor',
                CURLE_OPERATION_TIMEDOUT => 'Tempo limite de conexão excedido',
                CURLE_SSL_CONNECT_ERROR => 'Erro na conexão SSL/TLS',
                CURLE_GOT_NOTHING => 'Servidor não enviou resposta',
            ];

            $message = $errorMessages[$errno] ?? "Erro de conexão: {$error}";

            Response::json([
                'status' => 'error',
                'message' => $message,
                'data' => [
                    'http_code' => 0,
                    'error' => $error,
                    'error_code' => $errno,
                    'response_time' => $totalTime
                ]
            ]);
        }

        // Determinar mensagem baseada no código HTTP
        $statusMessages = [
            200 => 'OK - Site funcionando corretamente',
            201 => 'Created - Recurso criado',
            301 => 'Moved Permanently - Redirecionamento permanente',
            302 => 'Found - Redirecionamento temporário',
            400 => 'Bad Request - Requisição inválida',
            401 => 'Unauthorized - Autenticação necessária',
            403 => 'Forbidden - Acesso negado',
            404 => 'Not Found - Página não encontrada',
            500 => 'Internal Server Error - Erro interno do servidor',
            502 => 'Bad Gateway - Gateway inválido',
            503 => 'Service Unavailable - Serviço indisponível',
            504 => 'Gateway Timeout - Tempo limite do gateway'
        ];

        $message = $statusMessages[$httpCode] ?? "Código HTTP: {$httpCode}";

        Response::json([
            'status' => $httpCode === 200 ? 'success' : 'warning',
            'data' => [
                'http_code' => $httpCode,
                'message' => $message,
                'response_time' => $totalTime,
                'url' => $url
            ]
        ]);
    }
}
