/**
 * API Tokens Module
 * Gerenciamento de APIs externas (DETRAN, SEFAZ, etc)
 */
const ApiTokens = {
    // Dados carregados
    data: [],
    
    // Modal de edição
    editModal: null,
    currentEditId: null,
    
    /**
     * Carrega lista de APIs
     */
    async load() {
        console.log('[ApiTokens] Carregando APIs...');
        try {
            const response = await Api.get('api-tokens');
            console.log('[ApiTokens] Resposta:', response);
            
            if (response.status === 'success') {
                this.data = response.data || [];
                console.log('[ApiTokens] APIs carregadas:', this.data.length);
                this.renderTable();
                this.updateSelectors();
                this.updateCounter();
            }
        } catch (error) {
            console.error('[ApiTokens] Erro ao carregar:', error);
        }
    },
    
    /**
     * Carrega APIs apenas para preencher seletores (sem renderizar tabela)
     */
    async loadForSelectors() {
        try {
            const response = await Api.get('api-tokens');
            
            if (response.status === 'success') {
                this.data = response.data || [];
                this.updateSelectors();
                this.updateCounter();
            }
        } catch (error) {
            console.error('Erro ao carregar APIs para seletores:', error);
        }
    },
    
    /**
     * Atualiza contador no menu
     */
    updateCounter() {
        const counter = document.getElementById('api-tokens-count');
        if (counter) {
            counter.textContent = this.data.length;
        }
    },
    
    /**
     * Renderiza tabela de APIs
     */
    renderTable() {
        const tbody = document.getElementById('api-tokens-table-body');
        if (!tbody) return;
        
        if (this.data.length === 0) {
            tbody.innerHTML = `
                <tr>
                    <td colspan="6" class="text-center text-muted py-4">
                        <i class="fas fa-key fa-2x mb-2 d-block"></i>
                        Nenhuma API cadastrada.<br>
                        <small>Clique em "Nova API" para cadastrar.</small>
                    </td>
                </tr>
            `;
            return;
        }
        
        tbody.innerHTML = this.data.map(api => {
            const statusClass = api.last_test_status === 'success' ? 'success' : 
                               api.last_test_status === 'failed' ? 'danger' : 'secondary';
            const statusIcon = api.last_test_status === 'success' ? 'check-circle' :
                              api.last_test_status === 'failed' ? 'times-circle' : 'question-circle';
            
            return `
                <tr>
                    <td>
                        <strong>${this.escapeHtml(api.name)}</strong>
                        ${api.description ? `<br><small class="text-muted">${this.escapeHtml(api.description.substring(0, 50))}</small>` : ''}
                    </td>
                    <td>
                        <span class="badge bg-${api.api_type === 'gov' ? 'primary' : api.api_type === 'payment' ? 'success' : 'info'}">
                            ${api.api_type.toUpperCase()}
                        </span>
                    </td>
                    <td><code class="small">${api.token_masked || '***'}</code></td>
                    <td>
                        <span class="badge bg-${statusClass}">
                            <i class="fas fa-${statusIcon} me-1"></i>
                            ${api.last_test_status === 'success' ? 'OK' : api.last_test_status === 'failed' ? 'Falhou' : 'Não testado'}
                        </span>
                    </td>
                    <td>
                        <span class="badge ${api.sites_using > 0 ? 'bg-info' : 'bg-secondary'}">
                            ${api.sites_using || 0} site(s)
                        </span>
                    </td>
                    <td>
                        <button class="btn btn-sm btn-outline-primary me-1" onclick="ApiTokens.openEdit(${api.id})" title="Editar">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-success me-1" onclick="ApiTokens.test(${api.id})" title="Testar">
                            <i class="fas fa-vial"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" onclick="ApiTokens.delete(${api.id}, '${this.escapeHtml(api.name)}')" title="Excluir">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                </tr>
            `;
        }).join('');
    },
    
    /**
     * Atualiza seletores de API em outros formulários
     */
    updateSelectors() {
        const selectors = document.querySelectorAll('.api-token-selector');
        
        selectors.forEach(selector => {
            const currentValue = selector.value;
            
            selector.innerHTML = '<option value="">-- Nenhuma API --</option>';
            
            this.data.forEach(api => {
                if (api.is_active) {
                    selector.innerHTML += `
                        <option value="${api.id}" ${api.id == currentValue ? 'selected' : ''}>
                            ${this.escapeHtml(api.name)} (${api.api_type.toUpperCase()})
                        </option>
                    `;
                }
            });
        });
    },
    
    /**
     * Abre modal para criar nova API
     */
    openNew() {
        this.currentEditId = null;
        
        // Limpar formulário
        document.getElementById('api-token-id').value = '';
        document.getElementById('api-token-name').value = '';
        document.getElementById('api-token-description').value = '';
        document.getElementById('api-token-type').value = 'gov';
        document.getElementById('api-token-value').value = '';
        document.getElementById('api-token-base-url').value = '';
        document.getElementById('api-token-test-endpoint').value = '';
        document.getElementById('api-token-active').checked = true;
        
        // Atualizar título
        document.getElementById('modal-api-token-title').innerHTML = '<i class="fas fa-plus me-2"></i>Nova API';
        
        // Mostrar campo de token como obrigatório
        document.getElementById('api-token-value').required = true;
        document.getElementById('api-token-value-hint').textContent = 'Bearer Token ou chave de autenticação';
        
        // Abrir modal
        this.getModal().show();
    },
    
    /**
     * Abre modal para editar API existente
     */
    async openEdit(id) {
        this.currentEditId = id;
        UI.showLoading('Carregando...');
        
        try {
            const response = await Api.get(`api-token-item&id=${id}`);
            
            if (response.status !== 'success') {
                alert('Erro ao carregar API');
                return;
            }
            
            const api = response.data;
            
            // Preencher formulário
            document.getElementById('api-token-id').value = api.id;
            document.getElementById('api-token-name').value = api.name || '';
            document.getElementById('api-token-description').value = api.description || '';
            document.getElementById('api-token-type').value = api.api_type || 'gov';
            document.getElementById('api-token-value').value = api.token_value || '';
            document.getElementById('api-token-base-url').value = api.base_url || '';
            document.getElementById('api-token-test-endpoint').value = api.test_endpoint || '';
            document.getElementById('api-token-active').checked = api.is_active == 1;
            
            // Atualizar título
            document.getElementById('modal-api-token-title').innerHTML = 
                `<i class="fas fa-edit me-2"></i>Editar: ${this.escapeHtml(api.name)}`;
            
            // Token não é obrigatório ao editar (pode deixar em branco para manter)
            document.getElementById('api-token-value').required = false;
            document.getElementById('api-token-value-hint').textContent = 
                'Deixe em branco para manter o token atual';
            
            this.getModal().show();
            
        } catch (error) {
            console.error('Erro ao carregar API:', error);
            alert('Erro ao carregar dados da API');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Salva API (criar ou atualizar)
     */
    async save() {
        const form = document.getElementById('api-token-form');
        if (!form.checkValidity()) {
            form.reportValidity();
            return;
        }
        
        const name = document.getElementById('api-token-name').value.trim();
        const tokenValue = document.getElementById('api-token-value').value.trim();
        
        if (!name) {
            alert('Nome da API é obrigatório');
            return;
        }
        
        if (!this.currentEditId && !tokenValue) {
            alert('Token é obrigatório para criar nova API');
            return;
        }
        
        const payload = {
            name: name,
            description: document.getElementById('api-token-description').value.trim(),
            api_type: document.getElementById('api-token-type').value,
            base_url: document.getElementById('api-token-base-url').value.trim(),
            test_endpoint: document.getElementById('api-token-test-endpoint').value.trim(),
            is_active: document.getElementById('api-token-active').checked ? 1 : 0
        };
        
        // Só enviar token se foi preenchido
        if (tokenValue) {
            payload.token_value = tokenValue;
        }
        
        UI.showLoading('Salvando...');
        
        try {
            let response;
            
            if (this.currentEditId) {
                response = await Api.put(`api-token-item&id=${this.currentEditId}`, payload);
            } else {
                response = await Api.post('api-tokens', payload);
            }
            
            if (response.status === 'success') {
                this.getModal().hide();
                await this.load();
                alert(this.currentEditId ? 'API atualizada com sucesso!' : 'API criada com sucesso!');
            } else {
                alert('Erro: ' + (response.message || 'Erro desconhecido'));
            }
        } catch (error) {
            console.error('Erro ao salvar API:', error);
            alert('Erro ao salvar API');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Testa conexão com a API
     */
    async test(id) {
        UI.showLoading('Testando conexão...');
        
        try {
            const response = await Api.post(`api-token-test&id=${id}`, {});
            
            if (response.status === 'success') {
                const data = response.data;
                const statusColor = data.status === 'success' ? 'success' : 'danger';
                
                alert(
                    `🔗 Teste de Conexão\n\n` +
                    `URL: ${data.url_tested}\n` +
                    `HTTP Code: ${data.http_code}\n` +
                    `Tempo: ${data.response_time_ms}ms\n` +
                    `Status: ${data.status === 'success' ? '✅ Sucesso' : '❌ Falha'}\n` +
                    `${data.error ? `Erro: ${data.error}` : ''}`
                );
                
                // Recarregar para atualizar status
                await this.load();
            } else {
                alert('Erro: ' + (response.message || 'Erro ao testar'));
            }
        } catch (error) {
            console.error('Erro ao testar API:', error);
            alert('Erro ao testar conexão com a API');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Exclui uma API
     */
    async delete(id, name) {
        if (!confirm(`Excluir a API "${name}"?\n\nSites vinculados serão desvinculados automaticamente.`)) {
            return;
        }
        
        UI.showLoading('Excluindo...');
        
        try {
            const response = await Api.delete(`api-token-item&id=${id}`);
            
            if (response.status === 'success') {
                await this.load();
                
                // Recarregar sites também (podem ter sido desvinculados)
                if (typeof Sites !== 'undefined') {
                    await Sites.load();
                }
                
                alert(`API excluída com sucesso!${response.data?.sites_unlinked ? `\n${response.data.sites_unlinked} site(s) desvinculado(s).` : ''}`);
            } else {
                alert('Erro: ' + (response.message || 'Erro ao excluir'));
            }
        } catch (error) {
            console.error('Erro ao excluir API:', error);
            alert('Erro ao excluir API');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Obtém instância do modal
     */
    getModal() {
        if (!this.editModal) {
            this.editModal = new bootstrap.Modal(document.getElementById('modal-api-token'));
        }
        return this.editModal;
    },
    
    /**
     * Retorna API por ID
     */
    getById(id) {
        return this.data.find(api => api.id == id);
    },
    
    /**
     * Escape HTML
     */
    escapeHtml(text) {
        if (!text) return '';
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }
};

// Funções globais para compatibilidade
function openNewApiToken() { ApiTokens.openNew(); }
function saveApiToken() { ApiTokens.save(); }
