/**
 * Admin Panel - Authentication Module
 * @version 2.0.0
 */

const Auth = {
    _isLoading: false,

    /**
     * Fazer login
     */
    async login(username, password) {
        if (this._isLoading) return;
        this._isLoading = true;

        const btn = document.querySelector('#login-form .btn-login');
        const errorDiv = document.getElementById('login-error');

        UI.btnLoading(btn, true);
        UI.hide(errorDiv);

        try {
            const data = await Api.post('login', { username, password });

            if (data.status === 'success' && data.data) {
                AdminState.token = data.data.token;
                localStorage.setItem('admin_token', data.data.token);
                UI.toast('Login realizado com sucesso!', 'success');
                await this.showDashboard();
            } else {
                errorDiv.textContent = data.message || 'Credenciais inválidas';
                UI.show(errorDiv);
            }
        } catch (error) {
            errorDiv.textContent = 'Erro ao conectar com o servidor';
            UI.show(errorDiv);
        } finally {
            UI.btnLoading(btn, false);
            this._isLoading = false;
        }
    },

    /**
     * Fazer logout
     */
    logout() {
        AdminState.token = null;
        AdminState.currentUser = null;
        localStorage.removeItem('admin_token');
        
        UI.hide('#dashboard-page');
        UI.show('#login-page');
        
        // Limpar formulário
        document.getElementById('login-form')?.reset();
        UI.hide('#login-error');
    },

    /**
     * Carregar informações do usuário atual
     */
    async loadCurrentUser() {
        try {
            const data = await Api.get('me');

            if (data.status === 'success') {
                AdminState.currentUser = data.data;
                this.updateUIForUser();
                return data.data;
            }
        } catch (error) {
            // Silenciar em produção
        }
        return null;
    },

    /**
     * Atualizar UI baseado no tipo de usuário
     */
    updateUIForUser() {
        const user = AdminState.currentUser;
        if (!user) return;
        
        const isSuperAdmin = user.is_super_admin;
        
        // Mostrar/esconder elementos admin-only
        document.querySelectorAll('.admin-only').forEach(el => {
            el.style.display = isSuperAdmin ? '' : 'none';
        });
        
        // Mostrar/esconder elementos site-user-only
        document.querySelectorAll('.site-user-only').forEach(el => {
            el.style.display = !isSuperAdmin && user.site_id ? '' : 'none';
        });
        
        // Se for usuário de site, mostrar nome do site
        if (!isSuperAdmin && user.site_name) {
            UI.setText('#user-site-name', user.site_name);
        }
    },

    /**
     * Verificar se está autenticado
     */
    isAuthenticated() {
        return !!AdminState.token;
    },

    /**
     * Obter token atual
     */
    getToken() {
        return AdminState.token || localStorage.getItem('admin_token');
    },

    /**
     * Obter usuário atual
     */
    getCurrentUser() {
        return AdminState.currentUser;
    },

    /**
     * Mostrar dashboard após login
     * Delega para Dashboard.show() para evitar duplicação de lógica
     */
    async showDashboard() {
        // Usar Dashboard.show() que já tem toda a lógica correta
        if (typeof Dashboard !== 'undefined' && Dashboard.show) {
            await Dashboard.show();
        }
    },

    /**
     * Verificar autenticação no carregamento
     */
    async checkAuth() {
        if (this.isAuthenticated()) {
            // Esconder login imediatamente para evitar flash
            document.getElementById('login-page')?.classList.add('hidden');

            // Tentar até 2x antes de deslogar (protege contra erros temporários de rede/DB)
            for (let attempt = 0; attempt < 2; attempt++) {
                try {
                    const user = await this.loadCurrentUser();
                    if (user) {
                        await this.showDashboard();
                        return;
                    }
                } catch (error) {
                    console.warn(`Auth check attempt ${attempt + 1} failed:`, error.message);
                    if (attempt === 0) {
                        // Esperar 1s antes de tentar novamente
                        await new Promise(r => setTimeout(r, 1000));
                    }
                }
            }
            // Somente deslogar após 2 tentativas falharem
            this.logout();
        }
    },

    /**
     * Inicializar eventos de autenticação
     */
    init() {
        // Login form
        document.getElementById('login-form')?.addEventListener('submit', (e) => {
            e.preventDefault();
            const username = UI.getValue('#login-username');
            const password = UI.getValue('#login-password');
            this.login(username, password);
        });
        
        // Verificar autenticação existente
        this.checkAuth();
    }
};

// Expor globalmente
window.Auth = Auth;

// Funções de compatibilidade
window.logout = () => Auth.logout();
window.loadCurrentUser = () => Auth.loadCurrentUser();

// Variável global para compatibilidade
window.currentUserInfo = null;
Object.defineProperty(window, 'currentUserInfo', {
    get: () => AdminState.currentUser
});
