<?php
/**
 * Configuração do Banco de Dados MySQL
 * Admin Server Centralizado
 * 
 * IMPORTANTE: Este arquivo NUNCA deve ser acessível via web!
 */

// Bloquear acesso direto via navegador
if (basename($_SERVER['SCRIPT_FILENAME'] ?? '') === basename(__FILE__)) {
    http_response_code(403);
    die('Access denied');
}

// Configurações do banco de dados
define('DB_HOST', 'localhost');
define('DB_PORT', '3306');
define('DB_NAME', 'telas');
define('DB_USER', 'nexusintelligence');
define('DB_PASS', '@Araujo06');
define('DB_CHARSET', 'utf8mb4');

// Timezone padrão
date_default_timezone_set('America/Sao_Paulo');

/**
 * Capturar informações detalhadas do IP via ip-api.com
 * Garante IPv4 e metadados geográficos
 */
function getDetailedIPInfo(string $ip): array {
    // Se for IPv6 local, tenta ignorar ou tratar
    if ($ip === '::1' || $ip === '127.0.0.1' || $ip === 'localhost') {
        return ['query' => '127.0.0.1', 'status' => 'local', 'city' => 'Localhost'];
    }

    // Se o IP contiver ':', é um IPv6. Tenta tratar ou usar endpoint v4 do ip-api
    $baseUrl = "http://ip-api.com/json/";
    if (strpos($ip, ':') !== false) {
        // Para IPv6, o ip-api retorna os dados do IPv6. 
        // Não há como "descobrir" o IPv4 do cliente no server-side se ele conectou via v6.
    }

    $url = $baseUrl . $ip . "?fields=status,message,country,countryCode,region,regionName,city,zip,lat,lon,timezone,isp,org,as,query";
    
    $ctx = stream_context_create(['http' => ['timeout' => 2]]); // Timeout curto para não travar
    $response = @file_get_contents($url, false, $ctx);
    
    if ($response) {
        $data = json_decode($response, true);
        if ($data && $data['status'] === 'success') {
            return $data;
        }
    }
    
    return ['query' => $ip, 'status' => 'fail'];
}

// Chave de criptografia (MUDE ISSO EM PRODUÇÃO!)
define('ENCRYPTION_KEY', 'sua_chave_secreta_muito_segura_32chars!');
define('HMAC_KEY', 'sua_chave_hmac_secreta_aqui_32chars!!');

/**
 * Conectar ao banco de dados
 * OTIMIZADO: Conexão persistente para reduzir latência
 */
function getDBConnection(): PDO {
    static $pdo = null;
    
    if ($pdo === null) {
        try {
            $dsn = "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
            $options = [
                PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
                PDO::ATTR_EMULATE_PREPARES => false,
                // OTIMIZAÇÃO: Conexão persistente - reutiliza conexões existentes
                PDO::ATTR_PERSISTENT => true,
                // OTIMIZAÇÃO: Timeout de conexão
                PDO::ATTR_TIMEOUT => 10,
                // FORÇAR Timezone Brasília no MySQL (-03:00)
                PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES utf8mb4 COLLATE utf8mb4_unicode_ci, time_zone = '-03:00', sql_mode=(SELECT REPLACE(@@sql_mode,'ONLY_FULL_GROUP_BY',''))"
            ];
            
            $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);
        } catch (PDOException $e) {
            http_response_code(500);
            die(json_encode(['status' => 'error', 'message' => 'Database connection failed: ' . $e->getMessage()]));
        }
    }
    
    return $pdo;
}

/**
 * Funções de Criptografia
 */
function encryptData(string $data): string {
    $iv = openssl_random_pseudo_bytes(16);
    $encrypted = openssl_encrypt($data, 'AES-256-CBC', ENCRYPTION_KEY, 0, $iv);
    return base64_encode($iv . '::' . $encrypted);
}

function decryptData(string $encryptedData): string {
    $parts = explode('::', base64_decode($encryptedData), 2);
    if (count($parts) !== 2) return '';
    list($iv, $encrypted) = $parts;
    return openssl_decrypt($encrypted, 'AES-256-CBC', ENCRYPTION_KEY, 0, $iv);
}

/**
 * Gerar HMAC para validação
 */
function generateHMAC(string $data): string {
    return hash_hmac('sha256', $data, HMAC_KEY);
}

function verifyHMAC(string $data, string $hmac): bool {
    return hash_equals(generateHMAC($data), $hmac);
}

/**
 * Gerar token seguro
 */
function generateSecureToken(int $length = 64): string {
    return bin2hex(random_bytes($length / 2));
}

/**
 * Gerar API Key para sites
 */
function generateApiKey(): string {
    return 'sk_' . bin2hex(random_bytes(24));
}

/**
 * Validar API Key
 */
function validateApiKey(string $apiKey): ?array {
    $pdo = getDBConnection();
    $stmt = $pdo->prepare("
        SELECT s.*, 
               (SELECT COUNT(*) FROM events WHERE site_id = s.id AND DATE(created_at) = CURDATE()) as events_today
        FROM sites s 
        WHERE s.api_key = ? AND s.is_active = 1
        LIMIT 1
    ");
    $stmt->execute([$apiKey]);
    return $stmt->fetch() ?: null;
}

/**
 * Log de auditoria
 */
function logAudit(string $action, ?int $siteId = null, ?string $details = null, ?string $ip = null): void {
    try {
        $pdo = getDBConnection();
        $stmt = $pdo->prepare("
            INSERT INTO audit_logs (site_id, action, details, ip_address, created_at)
            VALUES (?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$siteId, $action, $details, $ip ?? $_SERVER['REMOTE_ADDR'] ?? '']);
    } catch (Exception $e) {
        // Silenciar erros de log
    }
}
