<?php
declare(strict_types=1);

header('Content-Type: application/json');

session_start();

// Verificar se está logado
if (empty($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Não autorizado']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Método não permitido']);
    exit;
}

try {
    $dbPath = __DIR__ . '/admin.db';
    $pdo = new PDO('sqlite:' . $dbPath);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Ler dados do request
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'JSON inválido']);
        exit;
    }

    $currentPassword = $data['current_password'] ?? '';
    $newPassword = $data['new_password'] ?? '';

    // Validar campos
    if (empty($currentPassword) || empty($newPassword)) {
        echo json_encode(['success' => false, 'message' => 'Preencha todos os campos']);
        exit;
    }

    if (strlen($newPassword) < 4) {
        echo json_encode(['success' => false, 'message' => 'A nova senha deve ter pelo menos 4 caracteres']);
        exit;
    }

    // Buscar hash atual
    $stmt = $pdo->prepare('SELECT value FROM settings WHERE key = :key LIMIT 1');
    $stmt->execute([':key' => 'admin_password_hash']);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row || !isset($row['value'])) {
        echo json_encode(['success' => false, 'message' => 'Senha não configurada no sistema']);
        exit;
    }

    $currentHash = $row['value'];

    // Verificar senha atual
    if (!password_verify($currentPassword, $currentHash)) {
        echo json_encode(['success' => false, 'message' => 'Senha atual incorreta']);
        exit;
    }

    // Gerar novo hash
    $newHash = password_hash($newPassword, PASSWORD_DEFAULT);

    // Atualizar no banco
    $stmt = $pdo->prepare('UPDATE settings SET value = :value, updated_at = CURRENT_TIMESTAMP WHERE key = :key');
    $stmt->execute([':key' => 'admin_password_hash', ':value' => $newHash]);

    if ($stmt->rowCount() === 0) {
        // Se não atualizou, tenta inserir
        $stmt = $pdo->prepare('INSERT INTO settings (key, value, updated_at) VALUES (:key, :value, CURRENT_TIMESTAMP)');
        $stmt->execute([':key' => 'admin_password_hash', ':value' => $newHash]);
    }

    echo json_encode(['success' => true, 'message' => 'Senha alterada com sucesso']);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Erro interno: ' . $e->getMessage()]);
}
