<?php
/**
 * Index - Página Principal com Proteções de Segurança
 * Site MG - Consulta IPVA Minas Gerais
 * 
 * Este arquivo serve o index.html aplicando as proteções de segurança
 * configuradas no painel admin.
 * 
 * @version 2.0.0
 */

declare(strict_types=1);

// Carrega o cliente de segurança
require_once __DIR__ . '/security-client.php';

// Identificador do site
$siteId = 'mg_site_main';

// Aplica proteções - bloqueia se necessário
$securityResult = SecurityClient::protect($siteId, true); // true = page mode (não API)

// Se chegou aqui, passou nas verificações
// Obtém a configuração de segurança para injetar o JS
$securityConfig = SecurityClient::getConfig($siteId);

// Lê o conteúdo do index.html
$htmlContent = file_get_contents(__DIR__ . '/index.html');

if ($htmlContent === false) {
    http_response_code(500);
    die('Erro ao carregar a página');
}

// === INJEÇÃO DE SCRIPTS DE SEGURANÇA ===

$securityScripts = '';

// 1. Bloqueio de DevTools/F12 (se ativo)
if ($securityConfig && !empty($securityConfig['block_devtools'])) {
    $securityScripts .= SecurityClient::getDevToolsProtectionJS() . "\n";
}

// 2. Injeta configurações de segurança como JSON para uso pelo frontend
$securityScripts .= '<script id="security-config" type="application/json">' . json_encode([
    'anti_bot' => $securityConfig['antibot_enabled'] ?? false,
    'rate_limit_enabled' => $securityConfig['rate_limit_enabled'] ?? false,
    'site_id' => $siteId
]) . '</script>' . "\n";

// 3. Script auxiliar para reportar eventos de segurança
$securityScripts .= <<<SCRIPT
<script>
// Função para reportar eventos de segurança ao servidor
window.reportSecurityEvent = function(eventType, details = {}) {
    const data = {
        site_id: '{$siteId}',
        event_type: eventType,
        details: details,
        timestamp: new Date().toISOString(),
        url: window.location.href,
        user_agent: navigator.userAgent
    };
    
    // Envia via track.php local (invisível no DevTools do usuário)
    fetch('track.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ action: eventType, metadata: { ...details, site_id: '{$siteId}' } }),
        keepalive: true
    }).catch(() => {});
};

// Reporta acesso à página inicial
document.addEventListener('DOMContentLoaded', function() {
    window.reportSecurityEvent('page_view', { page: 'index' });
});
</script>
SCRIPT;

// Meta tag de segurança
$securityMeta = '<meta name="security" content="protected">';

// === INJEÇÃO NO HTML ===

// Injeta meta tag no head
if (strpos($htmlContent, '<head>') !== false) {
    $htmlContent = str_replace('<head>', '<head>' . "\n" . $securityMeta, $htmlContent);
}

// Injeta scripts antes do </head>
if (strpos($htmlContent, '</head>') !== false) {
    $htmlContent = str_replace('</head>', $securityScripts . '</head>', $htmlContent);
} else {
    // Se não encontrar </head>, injeta no início do body
    $htmlContent = str_replace('<body', $securityScripts . '<body', $htmlContent);
}

// === HEADERS DE SEGURANÇA ===
header('Content-Type: text/html; charset=UTF-8');
header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Adiciona CSP básico se configurado
if ($securityConfig && !empty($securityConfig['csp_enabled'])) {
    $cspPolicy = $securityConfig['csp_policy'] ?? "default-src 'self' 'unsafe-inline' 'unsafe-eval' https:; img-src 'self' data: https:;";
    header("Content-Security-Policy: {$cspPolicy}");
}

// Output do HTML
echo $htmlContent;
