<?php
/**
 * Script para extrair CSS e JS inline dos arquivos HTML
 * Mantém o estilo original do site, apenas separa em arquivos
 */

$arquivos = ['index.html', 'debitos.html'];
$cssDir = __DIR__ . '/assets/css/';
$jsDir = __DIR__ . '/assets/js/';

// Criar diretórios se não existirem
if (!is_dir($cssDir)) mkdir($cssDir, 0755, true);
if (!is_dir($jsDir)) mkdir($jsDir, 0755, true);

foreach ($arquivos as $arquivo) {
    $caminhoArquivo = __DIR__ . '/' . $arquivo;
    
    if (!file_exists($caminhoArquivo)) {
        echo "Arquivo não encontrado: $arquivo\n";
        continue;
    }
    
    echo "\n========================================\n";
    echo "Processando: $arquivo\n";
    echo "========================================\n";
    
    $html = file_get_contents($caminhoArquivo);
    $nomeBase = pathinfo($arquivo, PATHINFO_FILENAME);
    
    // ========================================
    // EXTRAIR CSS INLINE (<style>...</style>)
    // ========================================
    $cssCount = 0;
    $cssTotal = '';
    
    // Padrão para capturar blocos <style>
    if (preg_match_all('/<style[^>]*>(.*?)<\/style>/is', $html, $matches)) {
        foreach ($matches[1] as $cssContent) {
            $cssContent = trim($cssContent);
            if (!empty($cssContent) && strlen($cssContent) > 50) {
                $cssTotal .= "/* ===== Bloco CSS " . (++$cssCount) . " ===== */\n";
                $cssTotal .= $cssContent . "\n\n";
            }
        }
    }
    
    if (!empty($cssTotal)) {
        $cssFile = $cssDir . $nomeBase . '.css';
        file_put_contents($cssFile, $cssTotal);
        echo "✓ CSS extraído: assets/css/{$nomeBase}.css (" . strlen($cssTotal) . " bytes, $cssCount blocos)\n";
    }
    
    // ========================================
    // EXTRAIR JS INLINE (<script>...</script>)
    // ========================================
    $jsCount = 0;
    $jsTotal = '';
    $jsApp = ''; // JS específico da aplicação (não bibliotecas)
    
    // Padrão para capturar blocos <script> sem src
    if (preg_match_all('/<script(?![^>]*\bsrc\b)[^>]*>(.*?)<\/script>/is', $html, $matches)) {
        foreach ($matches[1] as $jsContent) {
            $jsContent = trim($jsContent);
            if (!empty($jsContent) && strlen($jsContent) > 50) {
                // Verificar se é biblioteca conhecida ou código da aplicação
                $isLibrary = false;
                $libraryPatterns = [
                    '/^[\s\S]*jQuery\s*v[\d\.]+/i',
                    '/^\s*\(function\s*\(\s*factory\s*\)/i',
                    '/^\/\*!\s*Bootstrap/i',
                    '/qrcode/i',
                    '/flasher/i',
                ];
                
                foreach ($libraryPatterns as $pattern) {
                    if (preg_match($pattern, $jsContent)) {
                        $isLibrary = true;
                        break;
                    }
                }
                
                $jsCount++;
                
                if ($isLibrary || strlen($jsContent) > 50000) {
                    // Salvar bibliotecas grandes separadamente
                    $libName = 'lib_' . $jsCount;
                    
                    // Tentar identificar o nome da biblioteca
                    if (stripos($jsContent, 'jQuery') !== false && stripos($jsContent, 'v3') !== false) {
                        $libName = 'jquery.min';
                    } elseif (stripos($jsContent, 'Bootstrap') !== false) {
                        $libName = 'bootstrap.min';
                    } elseif (stripos($jsContent, 'qrcode') !== false || stripos($jsContent, 'QRCode') !== false) {
                        $libName = 'qrcode.min';
                    } elseif (stripos($jsContent, 'flasher') !== false) {
                        $libName = 'flasher.min';
                    }
                    
                    $libFile = $jsDir . $libName . '.js';
                    if (!file_exists($libFile)) {
                        file_put_contents($libFile, $jsContent);
                        echo "✓ Biblioteca extraída: assets/js/{$libName}.js (" . strlen($jsContent) . " bytes)\n";
                    }
                } else {
                    // Código da aplicação
                    $jsApp .= "/* ===== Bloco JS " . $jsCount . " ===== */\n";
                    $jsApp .= $jsContent . "\n\n";
                }
            }
        }
    }
    
    if (!empty($jsApp)) {
        $jsFile = $jsDir . $nomeBase . '.js';
        file_put_contents($jsFile, $jsApp);
        echo "✓ JS App extraído: assets/js/{$nomeBase}.js (" . strlen($jsApp) . " bytes)\n";
    }
    
    echo "Total de blocos processados: CSS=$cssCount, JS=$jsCount\n";
}

// ========================================
// LISTAR RECURSOS EXTERNOS (src, href)
// ========================================
echo "\n========================================\n";
echo "Recursos externos encontrados:\n";
echo "========================================\n";

foreach ($arquivos as $arquivo) {
    $caminhoArquivo = __DIR__ . '/' . $arquivo;
    if (!file_exists($caminhoArquivo)) continue;
    
    $html = file_get_contents($caminhoArquivo);
    echo "\n--- $arquivo ---\n";
    
    // Links CSS externos
    if (preg_match_all('/href=["\']([^"\']+\.css[^"\']*)["\']/', $html, $matches)) {
        echo "CSS externos:\n";
        foreach (array_unique($matches[1]) as $css) {
            if (strpos($css, 'data:') === false) {
                echo "  - $css\n";
            }
        }
    }
    
    // Scripts externos
    if (preg_match_all('/src=["\']([^"\']+\.js[^"\']*)["\']/', $html, $matches)) {
        echo "JS externos:\n";
        foreach (array_unique($matches[1]) as $js) {
            if (strpos($js, 'data:') === false) {
                echo "  - $js\n";
            }
        }
    }
}

echo "\n✅ Extração concluída!\n";
echo "Arquivos salvos em:\n";
echo "  - assets/css/\n";
echo "  - assets/js/\n";
