const { Server } = require("@modelcontextprotocol/sdk/server/index.js");
const { StdioServerTransport } = require("@modelcontextprotocol/sdk/server/stdio.js");
const {
  CallToolRequestSchema,
  ListToolsRequestSchema,
} = require("@modelcontextprotocol/sdk/types.js");
const axios = require("axios");
const { consultarMS, consultarMG } = require("./lib/placa_utils.js");
require("dotenv").config();

const server = new Server(
  {
    name: "mcp-utils-server",
    version: "1.0.0",
  },
  {
    capabilities: {
      tools: {},
    },
  }
);

/**
 * Ferramenta de Busca (Placeholder para integração com Google/Serper/etc)
 */
async function handleWebSearch(query) {
  // Exemplo de integração futura ou busca interna
  return {
    content: [
      {
        type: "text",
        text: `Busca realizada para: ${query}. (Integração real depende de chave de API em .env)`,
      },
    ],
  };
}

/**
 * Ferramenta para Gerar/Retornar Proxy
 */
async function handleGenerateProxy(country = "any") {
  // Baseado nos dados encontrados em mg/test-connection.php
  const proxyHost = process.env.PROXY_HOST || "gate.nodemaven.com:8080";
  const proxyAuth = process.env.PROXY_AUTH || "porto2k3_gmail_com-country-any-ipv4-true:chezc140hy";
  
  const proxyUrl = `http://${proxyAuth}@${proxyHost}`;
  
  return {
    content: [
      {
        type: "text",
        text: proxyUrl,
      },
    ],
  };
}

/**
 * Ferramenta para Resolver Captcha (Integração com CapSolver)
 */
async function handleSolveCaptcha(siteKey, pageUrl, type = "ReCaptchaV2TaskProxyLess") {
  const apiKey = process.env.CAPSOLVER_API_KEY;
  if (!apiKey) {
    throw new Error("CAPSOLVER_API_KEY não configurada no .env");
  }

  try {
    const payload = {
      clientKey: apiKey,
      task: {
        type: type,
        websiteURL: pageUrl,
        websiteKey: siteKey,
      },
    };

    const response = await axios.post("https://api.capsolver.com/createTask", payload);
    
    if (response.data.errorId !== 0) {
      throw new Error(`Erro ao criar tarefa CapSolver: ${response.data.errorCode} - ${response.data.errorDescription}`);
    }

    const taskId = response.data.taskId;
    
    return {
      content: [
        {
          type: "text",
          text: `Tarefa de captcha criada. Task ID: ${taskId}. Use check_captcha_status para consultar o resultado.`,
        },
      ],
    };
  } catch (error) {
    return {
      isError: true,
      content: [{ type: "text", text: error.message }],
    };
  }
}

server.setRequestHandler(ListToolsRequestSchema, async () => {
  return {
    tools: [
      {
        name: "web_search",
        description: "Realiza uma busca na web por informações atualizadas",
        inputSchema: {
          type: "object",
          properties: {
            query: { type: "string", description: "O termo de busca" },
          },
          required: ["query"],
        },
      },
      {
        name: "generate_proxy",
        description: "Gera uma URL de proxy formatada para uso em requests",
        inputSchema: {
          type: "object",
          properties: {
            country: { type: "string", description: "Código do país (ex: br, us, any)", default: "br" },
          },
        },
      },
      {
        name: "solve_captcha",
        description: "Envia um captcha para ser resolvido pelo CapSolver",
        inputSchema: {
          type: "object",
          properties: {
            siteKey: { type: "string", description: "Site key do captcha" },
            pageUrl: { type: "string", description: "URL da página onde está o captcha" },
            type: { type: "string", description: "Tipo do captcha no CapSolver (ReCaptchaV2TaskProxyLess, TurnstileTaskProxyLess, etc)", default: "ReCaptchaV2TaskProxyLess" },
          },
          required: ["siteKey", "pageUrl"],
        },
      },
      {
        name: "check_captcha_status",
        description: "Consulta o resultado de um captcha enviado ao CapSolver",
        inputSchema: {
          type: "object",
          properties: {
            taskId: { type: "string", description: "O ID da tarefa retornado por solve_captcha" },
          },
          required: ["taskId"],
        },
      },
      {
        name: "placa_lookup_ms",
        description: "Realiza consulta de débitos veiculares no Mato Grosso do Sul",
        inputSchema: {
          type: "object",
          properties: {
            placa: { type: "string", description: "Placa do veículo" },
            renavam: { type: "string", description: "Renavam do veículo" },
          },
          required: ["placa", "renavam"],
        },
      },
      {
        name: "placa_lookup_mg",
        description: "Realiza consulta de débitos veiculares em Minas Gerais usando o Renavam",
        inputSchema: {
          type: "object",
          properties: {
            renavam: { type: "string", description: "Renavam do veículo" },
          },
          required: ["renavam"],
        },
      },
    ],
  };
});

server.setRequestHandler(CallToolRequestSchema, async (request) => {
  const { name, arguments: args } = request.params;

  switch (name) {
    case "web_search":
      return await handleWebSearch(args.query);
    case "generate_proxy":
      return await handleGenerateProxy(args.country);
    case "solve_captcha":
      return await handleSolveCaptcha(args.siteKey, args.pageUrl, args.type);
    case "check_captcha_status":
      return await handleCheckCaptchaStatus(args.taskId);
    case "check_captcha_status":
      return await handleCheckCaptchaStatus(args.taskId);
    case "placa_lookup_ms":
      const msResult = await consultarMS(args.placa, args.renavam);
      return {
        content: [{ type: "text", text: JSON.stringify(msResult) }],
      };
    case "placa_lookup_mg":
      const mgResult = await consultarMG(args.renavam);
      return {
        content: [{ type: "text", text: JSON.stringify(mgResult) }],
      };
    default:
      throw new Error(`Ferramenta desconhecida: ${name}`);
  }
});

/**
 * Consulta o status/resultado de um captcha no CapSolver
 */
async function handleCheckCaptchaStatus(taskId) {
  const apiKey = process.env.CAPSOLVER_API_KEY;
  if (!apiKey) {
    throw new Error("CAPSOLVER_API_KEY não configurada no .env");
  }

  try {
    const payload = {
      clientKey: apiKey,
      taskId: taskId,
    };

    const response = await axios.post("https://api.capsolver.com/getTaskResult", payload);
    
    if (response.data.errorId !== 0) {
      throw new Error(`Erro ao consultar CapSolver: ${response.data.errorDescription}`);
    }

    if (response.data.status === "processing") {
      return {
        content: [{ type: "text", text: "STILL_WAITING: O captcha ainda está sendo resolvido." }],
      };
    }

    if (response.data.status === "ready") {
      return {
        content: [{ type: "text", text: response.data.solution.gRecaptchaResponse || response.data.solution.token }],
      };
    }

    throw new Error(`Status inesperado: ${response.data.status}`);
  } catch (error) {
    return {
      isError: true,
      content: [{ type: "text", text: error.message }],
    };
  }
}

async function main() {
  const transport = new StdioServerTransport();
  await server.connect(transport);
  console.error("MCP Utils Server running on stdio");
}

main().catch((error) => {
  console.error("Fatal error in main():", error);
  process.exit(1);
});
