<?php
/**
 * Security Init - Inicializador de Segurança para Páginas
 * 
 * Incluir no início de qualquer página PHP/HTML que precise de proteção.
 * Este arquivo aplica todas as verificações de segurança configuradas no admin.
 * 
 * Uso:
 * <?php require_once 'security-init.php'; ?>
 * 
 * @version 1.0.0
 */

// Remover headers que expõem informações
header_remove('X-Powered-By');
ini_set('expose_php', 'off');

// Carregar módulos necessários
require_once __DIR__ . '/client-config.php';
require_once __DIR__ . '/security-client.php';

// ============================================
// APLICAR PROTEÇÕES
// ============================================

try {
    SecurityClient::protect();
} catch (Exception $e) {
    // Log do erro mas não bloquear o usuário
    error_log("Security init error: " . $e->getMessage());
}

// ============================================
// FUNÇÕES PARA USO NAS PÁGINAS
// ============================================

/**
 * Retorna o JavaScript de proteção DevTools
 * Usar no HTML: <?= securityJS() ?>
 */
function securityJS(): string
{
    return SecurityClient::getDevToolsProtectionJS();
}

/**
 * Retorna configurações de segurança para uso no frontend (JSON)
 * Usar no JavaScript: const secConfig = <?= securityConfigJSON() ?>;
 */
function securityConfigJSON(): string
{
    $config = SecurityClient::getConfig();
    
    // Retornar apenas configurações relevantes para o frontend
    $frontendConfig = [
        'block_devtools' => (bool)($config['block_devtools'] ?? false),
        'block_right_click' => (bool)($config['block_right_click'] ?? false),
        'block_view_source' => (bool)($config['block_view_source'] ?? false),
        'block_copy' => (bool)($config['block_copy'] ?? false)
    ];
    
    return json_encode($frontendConfig);
}

/**
 * Verifica se deve mostrar captcha
 */
function shouldShowCaptcha(): bool
{
    $config = SecurityClient::getConfig();
    return (bool)($config['antibot_captcha_enabled'] ?? false);
}

/**
 * Obtém o token GOV configurado
 */
function getConfiguredGovToken(): ?string
{
    return SecurityClient::getGovToken();
}
