<?php
/**
 * API para trocar senha do admin
 * Timezone: America/Sao_Paulo
 */

date_default_timezone_set('America/Sao_Paulo');

require_once __DIR__ . '/require_auth.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

try {
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (!isset($data['current_password']) || !isset($data['new_password'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Senha atual e nova senha são obrigatórias']);
        exit;
    }

    $currentPassword = $data['current_password'];
    $newPassword = $data['new_password'];

    // Validações
    if (strlen($newPassword) < 6) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'A nova senha deve ter pelo menos 6 caracteres']);
        exit;
    }

    // Carregar configuração atual
    $configFile = __DIR__ . '/admin_data.json';
    if (!file_exists($configFile)) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Arquivo de configuração não encontrado']);
        exit;
    }

    $config = json_decode(file_get_contents($configFile), true);
    if (!$config || !isset($config['password'])) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Configuração inválida']);
        exit;
    }

    // Verificar senha atual
    if (!password_verify($currentPassword, $config['password'])) {
        http_response_code(401);
        echo json_encode(['status' => 'error', 'message' => 'Senha atual incorreta']);
        exit;
    }

    // Atualizar senha
    $config['password'] = password_hash($newPassword, PASSWORD_BCRYPT);
    $config['updated_at'] = date('Y-m-d H:i:s');

    // Salvar configuração
    if (file_put_contents($configFile, json_encode($config, JSON_PRETTY_PRINT)) === false) {
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => 'Erro ao salvar nova senha']);
        exit;
    }

    echo json_encode([
        'status' => 'success', 
        'message' => 'Senha alterada com sucesso',
        'updated_at' => $config['updated_at']
    ]);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erro no servidor: ' . $e->getMessage()]);
}
?>
