<?php
/**
 * API para consulta de débitos de veículo - DETRAN RS
 * Timezone: America/Sao_Paulo
 */
date_default_timezone_set('America/Sao_Paulo');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');

// Incluir client-config para buscar o token do servidor admin
require_once __DIR__ . '/client-config.php';

// Parâmetros da URL
$placa = isset($_GET['placa']) ? strtoupper(trim($_GET['placa'])) : null;
$renavam = isset($_GET['renavam']) ? trim($_GET['renavam']) : null;
$debug = isset($_GET['debug']) ? true : false;

if (!$placa || !$renavam) {
    echo json_encode([
        'success' => false,
        'error' => 'Parâmetros placa e renavam são obrigatórios',
        'exemplo' => 'api.php?placa=ABC1234&renavam=00330844717'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Buscar token de autenticação do servidor admin
$authToken = getDetranApiToken();
if (!$authToken) {
    echo json_encode([
        'success' => false,
        'error' => 'Token de autenticação não configurado. Contate o administrador.'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Função para fazer requisições cURL ao DETRAN RS
function consultarDetranRS($placa, $renavam, $authToken) {
    $ch = curl_init();
    
    // URL da API do DETRAN RS
    $url = "https://pcsdetran.procergs.com.br/pcsdetran/rest/veiculos/{$placa}/?renavam={$renavam}&contabiliza=false";
    
    // Configuração do Proxy
    $proxyHost = 'gw.dataimpulse.com';
    $proxyPort = 823;
    $proxyUser = '49ffd7ad05acca651768';
    $proxyPass = 'bfa2ac2adf4ef150';
    
    // Headers conforme o cURL fornecido - Token dinâmico do servidor admin
    $headers = [
        'User-Agent: Mozilla/5.0 (X11; Linux x86_64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/143.0.0.0 Safari/537.36',
        'Accept: application/json, text/plain, */*',
        'Accept-Encoding: gzip, deflate, br, zstd',
        'sec-ch-ua-platform: "Linux"',
        'Authorization: Bearer ' . $authToken,
        'sec-ch-ua: "Google Chrome";v="143", "Chromium";v="143", "Not A(Brand";v="24"',
        'sec-ch-ua-mobile: ?0',
        'Origin: https://pcsdetran.rs.gov.br',
        'Sec-Fetch-Site: cross-site',
        'Sec-Fetch-Mode: cors',
        'Sec-Fetch-Dest: empty',
        'Referer: https://pcsdetran.rs.gov.br/',
        'Accept-Language: pt-BR,pt;q=0.9'
    ];
    
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_ENCODING => '',
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_HTTPHEADER => $headers,
        // Configurações do Proxy
        CURLOPT_PROXY => $proxyHost,
        CURLOPT_PROXYPORT => $proxyPort,
        CURLOPT_PROXYUSERPWD => "{$proxyUser}:{$proxyPass}",
        CURLOPT_PROXYTYPE => CURLPROXY_HTTP
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'response' => $response,
        'httpCode' => $httpCode,
        'error' => $error
    ];
}

// Fazer a consulta
$resultado = consultarDetranRS($placa, $renavam, $authToken);

if ($resultado['error']) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro na requisição: ' . $resultado['error']
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if ($resultado['httpCode'] !== 200) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro HTTP: ' . $resultado['httpCode'],
        'response' => $resultado['response']
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Decodificar resposta JSON
$dadosDetran = json_decode($resultado['response'], true);

if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao decodificar JSON: ' . json_last_error_msg(),
        'response' => $resultado['response']
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Verificar se houve erro na resposta
if (isset($dadosDetran['identificacao']['temErro']) && $dadosDetran['identificacao']['temErro']) {
    echo json_encode([
        'success' => false,
        'error' => $dadosDetran['identificacao']['erro'] ?? 'Erro na consulta do veículo'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Montar dados do veículo
$identificacao = $dadosDetran['identificacao'] ?? [];
$licenciamento = $dadosDetran['licenciamento'] ?? [];
$imposto = $dadosDetran['imposto'] ?? [];
$seguro = $dadosDetran['seguro'] ?? [];
$infracao = $dadosDetran['infracao'] ?? [];
$restricao = $dadosDetran['restricao'] ?? [];
$expedicaoDocumento = $dadosDetran['expedicaoDocumento'] ?? [];

// Estruturar resposta
$dados = [
    'success' => true,
    'veiculo' => [
        'placa' => $identificacao['placa'] ?? '',
        'renavam' => $identificacao['renavam'] ?? '',
        'chassi' => $identificacao['chassi'] ?? '',
        'marca_modelo' => $identificacao['marcaModelo'] ?? '',
        'tipo' => $identificacao['tipo'] ?? '',
        'cor' => $identificacao['cor'] ?? '',
        'especie' => $identificacao['especie'] ?? '',
        'ano_fabricacao' => $identificacao['anoFabricacao'] ?? '',
        'ano_modelo' => $identificacao['anoModelo'] ?? '',
        'situacao' => $identificacao['situacao'] ?? '',
        'uf_placa' => $identificacao['ufPlaca'] ?? '',
        'municipio' => $identificacao['municipioRegistro'] ?? '',
        'cpf_proprietario' => $identificacao['cpfProprietario'] ?? '',
        'furtado' => $identificacao['furtado'] ?? false,
        'em_deposito' => $identificacao['emDeposito'] ?? false
    ],
    'licenciamento' => [
        'exercicio' => $licenciamento['exercicio'] ?? '',
        'situacao_documento' => $licenciamento['situacaoDocumento'] ?? '',
        'descricao_documento' => $licenciamento['descDocumento'] ?? '',
        'tem_crlve' => $licenciamento['temCrlve'] ?? false,
        'data_licenciamento' => $identificacao['dtLicenciamento'] ?? '',
        'vencimento_licenciamento' => $identificacao['dtVencLicenciamento'] ?? '',
        'valor_licenciamento' => isset($expedicaoDocumento['vlrLic']) ? 'R$ ' . number_format($expedicaoDocumento['vlrLic'], 2, ',', '.') : ''
    ],
    'debitos' => [],
    'total_debitos' => 'R$ 0,00'
];

// Processar débitos do IPVA
$totalDebitos = 0;

if (!empty($imposto['historico'])) {
    foreach ($imposto['historico'] as $historico) {
        // Só adicionar débitos pendentes ou que tenham débitos detalhados
        if ($historico['situacao'] === 'Pendente' || !empty($historico['debitos'])) {
            
            if (!empty($historico['debitos'])) {
                foreach ($historico['debitos'] as $debito) {
                    $valorOriginal = $debito['valorOriginal'] ?? '';
                    $valorComDesconto = $debito['valorTotalComDesconto'] ?? $valorOriginal;
                    
                    // Extrair valor numérico
                    $valorNumerico = floatval(str_replace(['.', ',', 'R$', ' '], ['', '.', '', ''], $valorComDesconto));
                    $totalDebitos += $valorNumerico;
                    
                    $dados['debitos'][] = [
                        'tipo' => 'IPVA ' . ($historico['exercicio'] ?? ''),
                        'exercicio' => $historico['exercicio'] ?? '',
                        'descricao' => $debito['descricao'] ?? '',
                        'valor_original' => $valorOriginal,
                        'valor_com_desconto' => $valorComDesconto,
                        'vencimento' => $historico['dataVencimento'] ?? '',
                        'situacao' => $historico['situacao'] ?? '',
                        'desconto_bom_motorista' => $debito['vlrDescontoBomMotorista'] ?? 'R$ 0,00',
                        'desconto_bom_cidadao' => $debito['vlrDescontoBomCidadao'] ?? 'R$ 0,00',
                        'desconto_antecipacao' => $debito['vlrDescontoAntecipacao'] ?? 'R$ 0,00',
                        'divida_ativa' => $historico['dividaAtiva'] ?? false
                    ];
                }
            } else {
                // Débito sem detalhamento (exercício pendente geral)
                $valorOriginal = $historico['valorOriginal'] ?? '';
                $valorNumerico = floatval(str_replace(['.', ',', 'R$', ' '], ['', '.', '', ''], $valorOriginal));
                $totalDebitos += $valorNumerico;
                
                $dados['debitos'][] = [
                    'tipo' => 'IPVA ' . ($historico['exercicio'] ?? ''),
                    'exercicio' => $historico['exercicio'] ?? '',
                    'descricao' => 'IPVA ' . ($historico['exercicio'] ?? ''),
                    'valor_original' => $valorOriginal,
                    'valor_com_desconto' => $valorOriginal,
                    'vencimento' => $historico['dataVencimento'] ?? '',
                    'situacao' => $historico['situacao'] ?? '',
                    'divida_ativa' => $historico['dividaAtiva'] ?? false
                ];
            }
        }
    }
}

// Adicionar valor do licenciamento se houver
if (!empty($expedicaoDocumento['vlrLic']) && $expedicaoDocumento['vlrLic'] > 0) {
    $totalDebitos += $expedicaoDocumento['vlrLic'];
    
    $dados['debitos'][] = [
        'tipo' => 'Licenciamento',
        'exercicio' => $expedicaoDocumento['exercRefLic'] ?? $licenciamento['exercicio'] ?? '',
        'descricao' => 'Licenciamento Anual',
        'valor_original' => 'R$ ' . number_format($expedicaoDocumento['vlrLic'], 2, ',', '.'),
        'valor_com_desconto' => 'R$ ' . number_format($expedicaoDocumento['vlrLic'], 2, ',', '.'),
        'vencimento' => $identificacao['dtVencLicenciamento'] ?? '',
        'situacao' => $expedicaoDocumento['txtSitLic'] ?? 'Pendente'
    ];
}

// Adicionar infrações se houver
if (!empty($infracao)) {
    // Infrações vencidas
    if (isset($infracao['qtVencidas']) && $infracao['qtVencidas'] > 0) {
        $valorInfracoes = $infracao['vlVencidas'] ?? 'R$ 0,00';
        $valorNumerico = floatval(str_replace(['.', ',', 'R$', ' '], ['', '.', '', ''], $valorInfracoes));
        $totalDebitos += $valorNumerico;
        
        $dados['debitos'][] = [
            'tipo' => 'Multas Vencidas',
            'descricao' => $infracao['qtVencidas'] . ' multa(s) vencida(s)',
            'valor_original' => $valorInfracoes,
            'valor_com_desconto' => $valorInfracoes,
            'situacao' => 'Vencida'
        ];
    }
    
    // Infrações a vencer
    if (isset($infracao['qtAVencer']) && $infracao['qtAVencer'] > 0) {
        $valorInfracoes = $infracao['vlAVencer'] ?? 'R$ 0,00';
        $valorNumerico = floatval(str_replace(['.', ',', 'R$', ' '], ['', '.', '', ''], $valorInfracoes));
        $totalDebitos += $valorNumerico;
        
        $dados['debitos'][] = [
            'tipo' => 'Multas a Vencer',
            'descricao' => $infracao['qtAVencer'] . ' multa(s) a vencer',
            'valor_original' => $valorInfracoes,
            'valor_com_desconto' => $valorInfracoes,
            'situacao' => 'A Vencer'
        ];
    }
}

// Adicionar seguro (DPVAT) se houver
if (!empty($seguro)) {
    $valorSeguroAtual = $seguro['valorExercAtual'] ?? 'R$ 0,00';
    $situacaoSeguro = $seguro['situacaoExercAtual'] ?? '';
    
    if ($situacaoSeguro !== 'Não devido' && $valorSeguroAtual !== 'R$ 0,00') {
        $valorNumerico = floatval(str_replace(['.', ',', 'R$', ' '], ['', '.', '', ''], $valorSeguroAtual));
        $totalDebitos += $valorNumerico;
        
        $dados['debitos'][] = [
            'tipo' => 'Seguro DPVAT',
            'exercicio' => $seguro['exercicioAtual'] ?? '',
            'descricao' => 'Seguro Obrigatório',
            'valor_original' => $valorSeguroAtual,
            'valor_com_desconto' => $valorSeguroAtual,
            'situacao' => $situacaoSeguro
        ];
    }
}

// Calcular total
$dados['total_debitos'] = 'R$ ' . number_format($totalDebitos, 2, ',', '.');

// Adicionar informações extras
$dados['restricoes'] = $restricao['restricoes'] ?? null;
$dados['infracoes_resumo'] = [
    'aguardando_defesa' => [
        'quantidade' => $infracao['qtAgPrazoDef'] ?? 0,
        'valor' => $infracao['vlAgPrazoDef'] ?? 'R$ 0,00'
    ],
    'aguardando_julgamento' => [
        'quantidade' => $infracao['qtAgPrazoJulg'] ?? 0,
        'valor' => $infracao['vlAgPrazoJulg'] ?? 'R$ 0,00'
    ],
    'a_vencer' => [
        'quantidade' => $infracao['qtAVencer'] ?? 0,
        'valor' => $infracao['vlAVencer'] ?? 'R$ 0,00'
    ],
    'vencidas' => [
        'quantidade' => $infracao['qtVencidas'] ?? 0,
        'valor' => $infracao['vlVencidas'] ?? 'R$ 0,00'
    ],
    'suspensas' => [
        'quantidade' => $infracao['qtSuspensas'] ?? 0,
        'valor' => $infracao['vlSuspensas'] ?? 'R$ 0,00'
    ]
];

// Adicionar histórico de IPVA (últimos pagamentos)
$dados['historico_ipva'] = [];
if (!empty($imposto['historico'])) {
    foreach ($imposto['historico'] as $historico) {
        $dados['historico_ipva'][] = [
            'exercicio' => $historico['exercicio'] ?? '',
            'situacao' => $historico['situacao'] ?? '',
            'vencimento' => $historico['dataVencimento'] ?? '',
            'valor_original' => $historico['valorOriginal'] ?? '',
            'divida_ativa' => $historico['dividaAtiva'] ?? false
        ];
    }
}

// Debug: adicionar resposta completa do DETRAN
if ($debug) {
    $dados['debug'] = [
        'url' => "https://pcsdetran.procergs.com.br/pcsdetran/rest/veiculos/{$placa}/?renavam={$renavam}&contabiliza=false",
        'http_code' => $resultado['httpCode'],
        'resposta_detran' => $dadosDetran
    ];
}

echo json_encode($dados, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
