const express = require('express');
const cors = require('cors');
const axios = require('axios');
const cheerio = require('cheerio');
const { wrapper } = require('axios-cookiejar-support');
const { CookieJar } = require('tough-cookie');

const app = express();
app.use(express.json());
app.use(cors());

// Configurações
const PORT = 3002; // Porta ajustada para 3002 para evitar conflito com BNMP (3001) e INSS (3000)
const API_KEY = 'b26b41804898eaee94c04a1795fc532ce5c30a6ea0cf9e1c0c525465774eabee';
const BASE_URL_CIPHERS = 'https://api-dh.ciphers.systems';

const sleep = (ms) => new Promise(resolve => setTimeout(resolve, ms));

async function consultarDespachante(placa, uf, renavam) {
    const jar = new CookieJar();
    const client = wrapper(axios.create({
        jar,
        withCredentials: true,
        baseURL: 'https://www.despachantedok.com.br'
    }));

    const result = {
        placa: placa.toUpperCase(),
        uf: uf ? uf.toUpperCase() : 'MG',
        renavam: renavam,
        status: 'erro',
        veiculo_dados: {}, // Dados básicos se conseguirmos extrair
        debitos: {
            multas: [],
            ipva: [],
            licenciamento: [],
            divida_ativa: [],
            outros: []
        },
        resumo: {
            total_multas: 0,
            total_ipva: 0,
            total_licenciamento: 0,
            total_geral: 0
        },
        meta: {
            orcamento_id: null,
            link_pagamento: null
        }
    };

    // UF inicial para a lógica
    let ufConsulta = result.uf;

    try {
        console.log(`[${placa}] Iniciando fluxo: 1. Procergs -> 2. Despachante`);

        // --- PASSO 1: Consulta Procergs (Prioritário para RS ou descoberta) ---
        // Tenta descobrir dados reais do veículo para guiar a consulta
        let dadosProcergs = null;
        try {
            console.log("   -> Consultando Procergs...");
            const resProcergs = await axios.get(`https://pcsdetran.procergs.com.br/pcsdetran/rest/veiculos/simplificada?placa=${placa.toUpperCase()}&renavam=${renavam || ''}&somenteRS=false`, {
                headers: { 'User-Agent': 'Mozilla/5.0' },
                timeout: 5000
            });

            if (resProcergs.data && !resProcergs.data.temErro) {
                dadosProcergs = resProcergs.data;
                console.log("   ✅ Procergs encontrou dados!");

                result.veiculo_dados = {
                    marca_modelo: dadosProcergs.marcaModelo,
                    ano_fabricacao: dadosProcergs.anoFabricacao,
                    ano_modelo: dadosProcergs.anoModelo,
                    cor: dadosProcergs.cor,
                    municipio: dadosProcergs.municipioRegistro,
                    uf: dadosProcergs.ufPlaca || ufConsulta, // Usa a UF do Procergs se disponível
                    situacao: dadosProcergs.situacao
                };

                // Atualiza a UF para a consulta do Despachante se o Procergs trouxer uma diferente
                if (dadosProcergs.ufPlaca && dadosProcergs.ufPlaca !== ufConsulta) {
                    console.log(`   ⚠️ Corrigindo UF de ${ufConsulta} para ${dadosProcergs.ufPlaca} (via Procergs)`);
                    ufConsulta = dadosProcergs.ufPlaca;
                    result.uf = ufConsulta;
                }
            } else {
                console.log("   ⚠️ Procergs: Veículo não localizado ou erro na resposta.");
            }
        } catch (eProcergs) {
            console.log("   ❌ Erro/Timeout Procergs (Pulando para Despachante).");
        }

        // --- PASSO 2: Despachante DOK ---
        console.log(`   -> Consultando Despachante DOK (${ufConsulta})...`);

        // 0. Home para CSRF
        const home = await client.get('/licenciamento-online', {
            headers: { 'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) Chrome/120.0.0.0 Safari/537.36' }
        });
        const $ = cheerio.load(home.data);
        const csrfToken = $('meta[name="csrf-token"]').attr('content');

        if (!csrfToken) throw new Error("Falha ao obter CSRF Token");

        const headers = {
            'X-CSRF-TOKEN': csrfToken,
            'Content-Type': 'application/json',
            'Accept': 'application/json',
            'X-Requested-With': 'XMLHttpRequest',
            'Origin': 'https://www.despachantedok.com.br',
            'Referer': 'https://www.despachantedok.com.br/licenciamento-online'
        };

        // 1. Validar Placa
        await client.post('/api/consulta-placa', {
            identificador: placa.toLowerCase(),
            tipo_identificador: "placa"
        }, { headers }).catch(e => console.log("Erro validação placa ignorado"));

        await sleep(1000);

        // 2. Gerar Orçamento
        // Nota: Email genérico para evitar bloqueio ou uso de dados pessoais
        const resOrcamento = await client.post('/api/consulta', {
            identificador: placa.toUpperCase(),
            email: "api-gateway@ciphers.systems",
            servico_tipo_nome: "licenciamento_online",
            uf: ufConsulta,
            renavam: renavam,
            documento_proprietario: ""
        }, { headers });

        if (!resOrcamento.data.success) {
            // Se falhar e a UF original for diferente da descoberta, tenta com a original como fallback?
            // Por enquanto, apenas lança erro
            throw new Error(resOrcamento.data.message || "Falha na criação do orçamento");
        }

        const dadosOrcamento = resOrcamento.data.data;
        result.meta.orcamento_id = dadosOrcamento.checkout_orcamento_id;
        result.meta.link_pagamento = dadosOrcamento.url_redirecionamento;

        // 3. Acessar Checkout para extrair dados detalhados
        const resCheckout = await client.get(dadosOrcamento.url_redirecionamento, {
            headers: { ...headers, 'Referer': 'https://www.despachantedok.com.br/' }
        });

        const htmlCheckout = resCheckout.data;
        // O JS com os dados geralmente está dentro de uma chamada receberDadosCheckout(...)
        const match = htmlCheckout.match(/receberDadosCheckout\((.*?)\)\s*"/);

        if (match && match[1]) {
            let rawJson = match[1].replace(/&quot;/g, '"');
            const dadosCompletos = JSON.parse(rawJson);

            result.status = 'sucesso';

            // Tentar extrair dados do veículo se disponíveis no JSON do checkout (as vezes vem em dados_servicos)
            if (dadosCompletos.dados_servicos) {
                // Mesclar com o que já temos do Procergs
                result.veiculo_dados = {
                    ...result.veiculo_dados,
                    cidade: dadosCompletos.dados_servicos.nome_cidade || result.veiculo_dados.cidade,
                    estado: dadosCompletos.dados_servicos.estado_sigla || result.veiculo_dados.estado,
                    documento_digital: dadosCompletos.dados_servicos.documento_digital
                };
            }

            // console.log("\n>>> DADOS DO VEÍCULO CONSOLIDADOS <<<");
            // console.log(JSON.stringify(result.veiculo_dados, null, 2));

            const itens = dadosCompletos.orcamento_itens || {};
            // console.log("\n>>> ITENS DO ORÇAMENTO (Despachante) <<<");
            // console.log(JSON.stringify(itens, null, 2));

            // Processar Multas (Vencer e Vencidas)
            const processarMultas = (listaItens) => {
                if(listaItens) {
                    listaItens.forEach(m => {
                        const valor = parseFloat(m.valor);
                        result.debitos.multas.push({
                            descricao: m.descricao || m.nome,
                            valor: valor,
                            vencimento: m.vencimento,
                            numero_auto: m.guia || m.aiip || m.codigo_barras || "N/D",
                            orgao_emissor: m.orgao || "DETRAN"
                        });
                        result.resumo.total_multas += valor;
                    });
                }
            };

            if (itens.multa_vencer) processarMultas(itens.multa_vencer.itens);
            if (itens.multa_vencida) processarMultas(itens.multa_vencida.itens);

            // Processar IPVA (Cota única, parcelado, anterior)
            Object.keys(itens).forEach(key => {
                if (key.includes('ipva') && itens[key].itens) {
                     itens[key].itens.forEach(i => {
                        const valor = parseFloat(i.valor);
                        result.debitos.ipva.push({
                            descricao: i.descricao || i.nome,
                            valor: valor,
                            ano: i.exercicio || null,
                            vencimento: i.vencimento
                        });
                        result.resumo.total_ipva += valor;
                     });
                }
            });

            // Processar Licenciamento
            // Processar Licenciamento e outros itens de forma mais genérica
            Object.entries(itens).forEach(([key, value]) => {
                if (value.itens) {
                    value.itens.forEach(i => {
                        const valor = parseFloat(i.valor);
                        const nome = (i.nome || "").toLowerCase();
                        const descricao = i.descricao || i.nome;

                        if (nome.includes('licenciamento')) {
                            result.debitos.licenciamento.push({
                                descricao: descricao,
                                valor: valor,
                                ano: i.exercicio || null
                            });
                            result.resumo.total_licenciamento += valor;
                        } else if (nome.includes('multa')) {
                            result.debitos.multas.push({
                                descricao: descricao,
                                valor: valor,
                                vencimento: i.vencimento,
                                numero_auto: i.guia || i.aiip || i.codigo_barras || "N/D",
                                orgao_emissor: i.orgao || "DETRAN"
                            });
                            result.resumo.total_multas += valor;
                        } else if (nome.includes('ipva')) {
                            result.debitos.ipva.push({
                                descricao: descricao,
                                valor: valor,
                                ano: i.exercicio || null,
                                vencimento: i.vencimento
                            });
                            result.resumo.total_ipva += valor;
                        } else if (nome.includes('divida_ativa') || key.includes('divida_ativa')) {
                            result.debitos.divida_ativa.push({ descricao: descricao, valor: valor });
                            result.resumo.total_geral += valor;
                        } else {
                            // Para outros itens não mapeados explicitamente
                            result.debitos.outros.push({ descricao: descricao, valor: valor, categoria: key });
                            result.resumo.total_geral += valor;
                        }
                    });
                }
            });

            // Dívida Ativa e Outros
            if (itens.divida_ativa && itens.divida_ativa.itens) {
                 itens.divida_ativa.itens.forEach(i => {
                    const valor = parseFloat(i.valor);
                    result.debitos.divida_ativa.push({ descricao: i.descricao, valor: valor });
                    result.resumo.total_geral += valor; // Soma no geral mas não tem resumo específico no objeto resumo inicial
                 });
            }

            // Total Geral Consolidado (Recalculado para garantir)
            result.resumo.total_geral = result.resumo.total_multas + result.resumo.total_ipva + result.resumo.total_licenciamento;
            // Se tiver outros débitos não mapeados explicitamente no resumo, adicionar
            // (A lógica original somava apenas os 3, mantive assim mas corrigi arredondamento)
            result.resumo.total_geral = parseFloat(result.resumo.total_geral.toFixed(2));

        } else {
            result.status = 'analise_manual';
            result.erro = "Não foi possível extrair os itens detalhados do HTML do checkout.";
        }

    } catch (e) {
        result.status = 'erro';
        result.erro = e.message;
        console.error(`Erro consulta [${placa}]:`, e.message);
    }

    return result;
}

// --- Proxy da Ciphers Systems (Mantido) ---
const proxyCiphers = async (req, res) => {
    try {
        const endpoint = req.path;
        const params = { ...req.query, apikey: API_KEY };
        const response = await axios.get(`${BASE_URL_CIPHERS}${endpoint}`, {
            params,
            headers: { 'Accept': 'application/json' },
            validateStatus: () => true
        });
        res.status(response.status).json(response.data);
    } catch (error) {
        res.status(500).json({ error: true, message: error.message });
    }
};

// --- ROTAS ---

// Nova rota POST conforme solicitado
app.post('/api/consultar-veiculo', async (req, res) => {
    const { placa, uf, renavam } = req.body;

    if (!placa || !renavam) {
        return res.status(400).json({ error: 'Placa e Renavam são obrigatórios' });
    }

    const dados = await consultarDespachante(placa, uf, renavam);
    res.json(dados);
});

// Manter rotas GET legado e Proxy
app.get('/api/v1/vehicle/debts', async (req, res) => {
    const { plate, uf, renavam } = req.query;
    if (!plate) return res.status(400).json({ error: "Placa é obrigatória" });
    const dados = await consultarDespachante(plate, uf, renavam);
    res.json(dados);
});

// Rotas Ciphers
app.get('/api/v2/serasa/zipcode', proxyCiphers);
app.get('/api/v1/fgts/offset', proxyCiphers);
app.get('/api/v1/serasa/phone', proxyCiphers);
app.get('/api/v2/generic/phone', proxyCiphers);
app.get('/api/v1/generic/phone', proxyCiphers);
app.get('/api/v1/serasa/name', proxyCiphers);
app.get('/api/v1/serasa/cpf', proxyCiphers);
app.get('/api/v2/generic/cpf', proxyCiphers);
app.get('/api/v1/basic/cpf', proxyCiphers);
app.get('/api/v1/vehicle/plate', proxyCiphers);
app.get('/api/v2/vehicle/plate', proxyCiphers);
app.get('/api/v2/serasa/family', proxyCiphers);
app.get('/api/v1/card/bin', proxyCiphers);
app.get('/api/v2/card/bin', proxyCiphers);
app.get('/api/v2/company/cnpj', proxyCiphers);
app.get('/api/v1/rj/cpf', proxyCiphers);
app.get('/api/v2/serasa/email', proxyCiphers);
app.get('/api/v1/generic/shortcpf', proxyCiphers);

app.get('/', (req, res) => res.json({ status: "online", endpoints: "Ciphers + Vehicle Debts" }));

app.listen(PORT, () => {
    console.log(`🚀 Servidor rodando na porta ${PORT}`);
    console.log(`Endpoint POST: http://localhost:${PORT}/api/consultar-veiculo`);
});