<?php
require_once __DIR__ . '/require_auth.php';

header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');
header('Expires: 0');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

date_default_timezone_set('America/Sao_Paulo');

function clip_str(string $value, int $maxLen): string {
    if ($maxLen <= 0) return '';
    if (function_exists('mb_substr')) {
        return (string)mb_substr($value, 0, $maxLen, 'UTF-8');
    }
    return substr($value, 0, $maxLen);
}

function get_default_pix_config() {
    return [
        // Modo: 'auto' (API) ou 'manual' (QR Code estático)
        'mode' => 'auto',
        
        // Configurações PIX Manual
        'pixKey' => '',
        'pixKeyType' => 'cpf', // cpf, cnpj, email, phone, random
        'receiverName' => 'PAGAMENTO',
        'receiverCity' => 'SAO PAULO',
        'amount' => null,
        
        // Configurações PIX Automático (API)
        'apiUrl' => 'https://api.raypaymentsv2.com.br/functions/v1',
        'secretKey' => '',
        'companyId' => '',
        
        // Controle
        'active' => true
    ];
}

function get_default_valores_meses() {
    return [
        '11/2025' => 75.90,
        '12/2025' => 75.90,
        '01/2026' => 75.90
    ];
}

try {
    $dbPath = __DIR__ . '/admin.db';
    $pdo = new PDO('sqlite:' . $dbPath);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);

    // Generic settings table (flexível para futuras configs)
    $pdo->exec("CREATE TABLE IF NOT EXISTS settings (
        key TEXT PRIMARY KEY,
        value TEXT NOT NULL,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )");

    // Determinar qual configuração está sendo solicitada
    $configType = isset($_GET['type']) ? $_GET['type'] : 'pix';

    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        
        if ($configType === 'valores_meses') {
            // Retornar configuração de valores dos meses
            $stmt = $pdo->prepare('SELECT value FROM settings WHERE key = :key LIMIT 1');
            $stmt->execute([':key' => 'valores_meses']);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            $cfg = get_default_valores_meses();
            if ($row && isset($row['value'])) {
                $decoded = json_decode($row['value'], true);
                if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                    $cfg = $decoded;
                }
            }

            echo json_encode(['status' => 'success', 'data' => $cfg]);
            exit;
        }
        
        // Configuração PIX padrão
        $stmt = $pdo->prepare('SELECT value FROM settings WHERE key = :key LIMIT 1');
        $stmt->execute([':key' => 'pix_config']);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        $cfg = get_default_pix_config();
        if ($row && isset($row['value'])) {
            $decoded = json_decode($row['value'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
                $cfg = array_merge($cfg, $decoded);
            }
        }

        echo json_encode(['status' => 'success', 'data' => $cfg]);
        exit;
    }

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = file_get_contents('php://input');
        $data = json_decode($input, true);
        if (json_last_error() !== JSON_ERROR_NONE || !is_array($data)) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Invalid JSON']);
            exit;
        }

        // Verificar se é para salvar valores dos meses
        if ($configType === 'valores_meses' || isset($data['valores_meses'])) {
            $valoresMeses = isset($data['valores_meses']) ? $data['valores_meses'] : $data;
            
            // Validar e normalizar valores
            $valoresNormalizados = [];
            foreach ($valoresMeses as $mesAno => $valor) {
                // Aceita número ou string ("75,90", "75.90")
                if (is_string($valor)) {
                    $normalized = str_replace(['.', ','], ['', '.'], $valor);
                    $valor = floatval($normalized);
                } else {
                    $valor = floatval($valor);
                }
                
                if ($valor >= 0) {
                    $valoresNormalizados[$mesAno] = $valor;
                }
            }

            $stmt = $pdo->prepare('INSERT INTO settings (key, value, updated_at) VALUES (:key, :value, CURRENT_TIMESTAMP)
                ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP');
            $stmt->execute([
                ':key' => 'valores_meses',
                ':value' => json_encode($valoresNormalizados, JSON_UNESCAPED_UNICODE)
            ]);

            echo json_encode(['status' => 'success', 'data' => $valoresNormalizados]);
            exit;
        }

        // Configuração PIX Completa
        $cfg = get_default_pix_config();

        // Modo PIX: auto ou manual
        $mode = isset($data['mode']) && in_array($data['mode'], ['auto', 'manual']) ? $data['mode'] : 'auto';
        $active = isset($data['active']) ? (bool)$data['active'] : true;

        // Configurações PIX Manual
        $pixKey = isset($data['pixKey']) ? trim((string)$data['pixKey']) : '';
        $pixKeyType = isset($data['pixKeyType']) && in_array($data['pixKeyType'], ['cpf', 'cnpj', 'email', 'phone', 'random']) 
            ? $data['pixKeyType'] : 'cpf';
        $receiverName = isset($data['receiverName']) ? trim((string)$data['receiverName']) : $cfg['receiverName'];
        $receiverCity = isset($data['receiverCity']) ? trim((string)$data['receiverCity']) : $cfg['receiverCity'];
        $amount = array_key_exists('amount', $data) ? $data['amount'] : null;

        // Configurações PIX Automático (API)
        $apiUrl = isset($data['apiUrl']) ? trim((string)$data['apiUrl']) : $cfg['apiUrl'];
        $secretKey = isset($data['secretKey']) ? trim((string)$data['secretKey']) : '';
        $companyId = isset($data['companyId']) ? trim((string)$data['companyId']) : '';

        // Validação baseada no modo
        if ($mode === 'manual' && $pixKey === '') {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Chave PIX é obrigatória no modo manual']);
            exit;
        }

        if ($mode === 'auto' && ($secretKey === '' || $companyId === '')) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Secret Key e Company ID são obrigatórios no modo automático']);
            exit;
        }

        // Constraints do BR Code
        if ($receiverName === '') $receiverName = $cfg['receiverName'];
        if ($receiverCity === '') $receiverCity = $cfg['receiverCity'];
        $receiverName = clip_str($receiverName, 25);
        $receiverCity = clip_str($receiverCity, 15);

        if ($amount === '' || $amount === null) {
            $amount = null;
        } else {
            // aceita número ou string ("299,99", "299.99")
            if (is_string($amount)) {
                $normalized = str_replace(['.', ','], ['', '.'], $amount);
                $amount = $normalized;
            }
            $num = floatval($amount);
            if (!is_finite($num) || $num <= 0) {
                http_response_code(400);
                echo json_encode(['status' => 'error', 'message' => 'amount is invalid']);
                exit;
            }
            $amount = $num;
        }

        $newCfg = [
            'mode' => $mode,
            'active' => $active,
            
            // PIX Manual
            'pixKey' => $pixKey,
            'pixKeyType' => $pixKeyType,
            'receiverName' => $receiverName,
            'receiverCity' => $receiverCity,
            'amount' => $amount,
            
            // PIX Automático
            'apiUrl' => $apiUrl,
            'secretKey' => $secretKey,
            'companyId' => $companyId
        ];

        $stmt = $pdo->prepare('INSERT INTO settings (key, value, updated_at) VALUES (:key, :value, CURRENT_TIMESTAMP)
            ON CONFLICT(key) DO UPDATE SET value = excluded.value, updated_at = CURRENT_TIMESTAMP');
        $stmt->execute([
            ':key' => 'pix_config',
            ':value' => json_encode($newCfg, JSON_UNESCAPED_UNICODE)
        ]);

        echo json_encode(['status' => 'success', 'data' => $newCfg]);
        exit;
    }

    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Internal Server Error', 'error' => $e->getMessage()]);
}
