<?php
/**
 * API Proxy para consulta de débitos de veículo - DETRAN MS
 * Chama a API remota em http://52.15.37.114/dk/api.php
 * Timezone: America/Sao_Paulo
 */
date_default_timezone_set('America/Sao_Paulo');

// Remover headers que expõem informações do servidor
header_remove('X-Powered-By');
ini_set('expose_php', 'off');

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Server: WebServer');

/**
 * Rate limiting simples por IP
 */
function checkRateLimit(): bool {
    $ip = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['HTTP_X_REAL_IP'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    if (strpos($ip, ',') !== false) $ip = trim(explode(',', $ip)[0]);
    
    $cacheDir = __DIR__ . '/cache';
    if (!is_dir($cacheDir)) @mkdir($cacheDir, 0755, true);
    
    $rateLimitFile = $cacheDir . '/api_rate_' . md5($ip) . '.json';
    $now = time();
    $windowStart = $now - 60;
    $requests = [];
    
    if (file_exists($rateLimitFile)) {
        $data = @json_decode(file_get_contents($rateLimitFile), true);
        if (is_array($data)) {
            $requests = array_filter($data, function($ts) use ($windowStart) {
                return $ts > $windowStart;
            });
        }
    }
    
    if (count($requests) >= 30) { // 30 consultas por minuto por IP
        return false;
    }
    
    $requests[] = $now;
    @file_put_contents($rateLimitFile, json_encode(array_values($requests)));
    return true;
}

// Verificar rate limit
if (!checkRateLimit()) {
    http_response_code(429);
    echo json_encode(['success' => false, 'error' => 'Muitas requisições. Aguarde um momento.'], JSON_UNESCAPED_UNICODE);
    exit;
}

// URL da API remota
$apiRemota = 'http://52.15.37.114/dk/api.php';

// Parâmetros - aceita GET ou JSON no body
$placa = null;
$renavam = null;
$debug = isset($_GET['debug']) ? true : false;

// Primeiro tenta GET
if (isset($_GET['placa']) && isset($_GET['renavam'])) {
    $placa = strtoupper(trim($_GET['placa']));
    $renavam = trim($_GET['renavam']);
} else {
    // Se não vier via GET, tenta JSON no body
    $jsonInput = file_get_contents('php://input');
    if ($jsonInput) {
        $data = json_decode($jsonInput, true);
        if ($data && isset($data['placa']) && isset($data['renavam'])) {
            $placa = strtoupper(trim($data['placa']));
            $renavam = trim($data['renavam']);
        }
    }
}

if (!$placa || !$renavam) {
    echo json_encode([
        'success' => false,
        'error' => 'Parâmetros placa e renavam são obrigatórios',
        'exemplo' => 'api.php?placa=ABC1234&renavam=00330844717'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Função para chamar a API remota
 */
function chamarApiRemota($url, $placa, $renavam) {
    $ch = curl_init();
    
    $urlCompleta = $url . '?placa=' . urlencode($placa) . '&renavam=' . urlencode($renavam);
    
    $curlOptions = [
        CURLOPT_URL => $urlCompleta,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_CONNECTTIMEOUT => 30,
        CURLOPT_HTTPHEADER => [
            'Accept: application/json',
            'User-Agent: MS-Proxy/1.0'
        ]
    ];
    
    curl_setopt_array($ch, $curlOptions);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    
    curl_close($ch);
    
    return [
        'response' => $response,
        'httpCode' => $httpCode,
        'error' => $error
    ];
}

// ==================== FLUXO PRINCIPAL ====================

// Chamar API remota
$resultado = chamarApiRemota($apiRemota, $placa, $renavam);

if ($resultado['error']) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao conectar com API remota: ' . $resultado['error']
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

if ($resultado['httpCode'] !== 200) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro HTTP da API remota: ' . $resultado['httpCode'],
        'response' => $debug ? $resultado['response'] : null
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Decodificar resposta
// Remover BOM (Byte Order Mark) se existir
$response = $resultado['response'];
$response = preg_replace('/^\xEF\xBB\xBF/', '', $response);
$response = trim($response);

$dadosDetran = json_decode($response, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode([
        'success' => false,
        'error' => 'Erro ao decodificar JSON: ' . json_last_error_msg(),
        'raw_response' => $response,
        'response_length' => strlen($response)
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// Se a API remota já retorna no formato correto, repassar diretamente
if (isset($dadosDetran['success'])) {
    echo json_encode($dadosDetran, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

// ==================== PROCESSAR DADOS ====================

$data = $dadosDetran['data'] ?? [];
$vehicleData = $data['vehicleData'] ?? [];
$licensing = $data['licensing'] ?? [];
$fines = $data['fines'] ?? [];
$ipva = $data['ipva'] ?? [];
$others = $data['others'] ?? null;
$subTotalDetran = $data['subTotalDetran'] ?? '0,00';
$totalGeneral = $data['totalGeneral'] ?? '0,00';

// Função para converter valor string para float
function parseValor($valor) {
    if (is_numeric($valor)) return floatval($valor);
    if (empty($valor)) return 0;
    // Remove "R$ ", pontos de milhar e converte vírgula em ponto
    $valor = str_replace(['R$', ' ', '.'], '', $valor);
    $valor = str_replace(',', '.', $valor);
    return floatval($valor);
}

// Estruturar resposta no formato padronizado (compatível com RS)
$dados = [
    'success' => true,
    'veiculo' => [
        'placa' => $vehicleData['plate'] ?? $placa,
        'renavam' => $vehicleData['renavam'] ?? $renavam,
        'chassi' => $vehicleData['chassi'] ?? '',
        'motor' => $vehicleData['motor'] ?? '',
        'marca_modelo' => $vehicleData['model'] ?? '',
        'tipo' => $vehicleData['category'] ?? '',
        'cor' => $vehicleData['color'] ?? '',
        'especie' => '',
        'ano_fabricacao' => $vehicleData['fabricationYear'] ?? '',
        'ano_modelo' => $vehicleData['modelYear'] ?? '',
        'situacao' => $vehicleData['licensing'] ?? '',
        'uf_placa' => 'MS',
        'municipio' => $vehicleData['city'] ?? '',
        'crlv_digital' => $vehicleData['crlvDigital'] ?? '',
        'exp_documento' => $vehicleData['expDoc'] ?? '',
        'observacoes' => $vehicleData['observations'] ?? []
    ],
    'licenciamento' => [
        'exercicio' => '',
        'situacao_documento' => $vehicleData['licensing'] ?? '',
        'vencimento_licenciamento' => $vehicleData['expDoc'] ?? '',
        'valor_licenciamento' => ''
    ],
    'debitos' => [],
    'total_debitos' => 'R$ 0,00',
    'historico_ipva' => [],
    'infracoes_resumo' => null
];

$totalDebitos = 0;

// Processar Licenciamento
if (!empty($licensing)) {
    $licLabel = $licensing['label'] ?? '';
    $licValue = $licensing['value'] ?? '0,00';
    $licValorNum = parseValor($licValue);
    
    if ($licValorNum > 0) {
        $totalDebitos += $licValorNum;
        $dados['debitos'][] = [
            'id' => uniqid('lic_'),
            'tipo' => 'Licenciamento',
            'exercicio' => '',
            'descricao' => $licLabel ?: 'Licenciamento',
            'valor' => $licValorNum,
            'valor_original' => 'R$ ' . $licValue,
            'valor_com_desconto' => 'R$ ' . $licValue,
            'vencimento' => '',
            'situacao' => 'Pendente'
        ];
        
        $dados['licenciamento']['valor_licenciamento'] = 'R$ ' . $licValue;
    }
}

// Processar IPVA
if (!empty($ipva) && isset($ipva['debts'])) {
    foreach ($ipva['debts'] as $item) {
        $valor = parseValor($item['value'] ?? '0');
        $label = $item['label'] ?? '';
        
        // Extrair ano do label (ex: "IPVA 2026" ou "IPVA ATRASO 2025")
        preg_match('/(\d{4})/', $label, $matches);
        $ano = $matches[1] ?? '';
        
        // Verificar se é IPVA em atraso
        $emAtraso = stripos($label, 'ATRASO') !== false;
        
        if ($valor > 0) {
            $totalDebitos += $valor;
            $dados['debitos'][] = [
                'id' => uniqid('ipva_'),
                'tipo' => 'IPVA ' . $ano,
                'exercicio' => $ano,
                'descricao' => $label,
                'valor' => $valor,
                'valor_original' => 'R$ ' . number_format($valor, 2, ',', '.'),
                'valor_com_desconto' => 'R$ ' . number_format($valor, 2, ',', '.'),
                'vencimento' => '',
                'situacao' => $emAtraso ? 'Em Atraso' : 'Pendente',
                'divida_ativa' => $emAtraso
            ];
            
            // Adicionar ao histórico IPVA
            $dados['historico_ipva'][] = [
                'exercicio' => $ano,
                'situacao' => $emAtraso ? 'Em Atraso' : 'Pendente',
                'valor_original' => 'R$ ' . number_format($valor, 2, ',', '.'),
                'vencimento' => '',
                'divida_ativa' => $emAtraso
            ];
        } else {
            // IPVA com valor 0 (quitado ou isento)
            $dados['historico_ipva'][] = [
                'exercicio' => $ano,
                'situacao' => 'Liquidado',
                'valor_original' => 'R$ 0,00',
                'vencimento' => '',
                'divida_ativa' => false
            ];
        }
    }
    
    // Adicionar total do IPVA se houver
    if (!empty($ipva['total'])) {
        $dados['ipva_total'] = 'R$ ' . $ipva['total'];
    }
}

// Processar Multas
if (!empty($fines) && isset($fines['debts'])) {
    $qtdMultas = 0;
    $valorMultas = 0;
    
    foreach ($fines['debts'] as $fine) {
        $valor = parseValor($fine['value'] ?? '0');
        $label = $fine['label'] ?? 'Multa';
        
        if ($valor > 0) {
            $totalDebitos += $valor;
            $qtdMultas++;
            $valorMultas += $valor;
            
            $dados['debitos'][] = [
                'id' => uniqid('multa_'),
                'tipo' => 'Multa',
                'descricao' => $label,
                'valor' => $valor,
                'valor_original' => 'R$ ' . number_format($valor, 2, ',', '.'),
                'valor_com_desconto' => 'R$ ' . number_format($valor, 2, ',', '.'),
                'vencimento' => '',
                'situacao' => 'Pendente'
            ];
        }
    }
    
    // Resumo de infrações
    if ($qtdMultas > 0 || !empty($fines['total'])) {
        $totalMultas = parseValor($fines['total'] ?? $valorMultas);
        $dados['infracoes_resumo'] = [
            'vencidas' => [
                'quantidade' => $qtdMultas,
                'valor' => 'R$ ' . number_format($totalMultas, 2, ',', '.')
            ],
            'a_vencer' => ['quantidade' => 0, 'valor' => 'R$ 0,00'],
            'suspensas' => ['quantidade' => 0, 'valor' => 'R$ 0,00'],
            'aguardando_defesa' => ['quantidade' => 0, 'valor' => 'R$ 0,00'],
            'aguardando_julgamento' => ['quantidade' => 0, 'valor' => 'R$ 0,00']
        ];
    }
}

// Processar Outros débitos
if (!empty($others)) {
    if (is_array($others)) {
        foreach ($others as $other) {
            $valor = parseValor($other['value'] ?? '0');
            if ($valor > 0) {
                $totalDebitos += $valor;
                $dados['debitos'][] = [
                    'id' => uniqid('outros_'),
                    'tipo' => 'Outros',
                    'descricao' => $other['label'] ?? 'Outros Débitos',
                    'valor' => $valor,
                    'valor_original' => 'R$ ' . number_format($valor, 2, ',', '.'),
                    'valor_com_desconto' => 'R$ ' . number_format($valor, 2, ',', '.'),
                    'vencimento' => '',
                    'situacao' => 'Pendente'
                ];
            }
        }
    }
}

// Total geral
$totalGenralNum = parseValor($totalGeneral);
if ($totalGenralNum > 0) {
    $totalDebitos = $totalGenralNum;
}

$dados['total_debitos'] = 'R$ ' . number_format($totalDebitos, 2, ',', '.');
$dados['sub_total_detran'] = 'R$ ' . $subTotalDetran;

// Adicionar dados brutos para debug
if ($debug) {
    $dados['debug'] = [
        'raw_response' => $dadosDetran
    ];
}

// Retornar JSON
echo json_encode($dados, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
